package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"

	"github.com/twitchtv/twirp"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	dclient "code.justin.tv/web/discovery/client"
)

func (s *Server) OnboardGame(ctx context.Context, params *rbacrpc.OnboardGameRequest) (*rbacrpc.OnboardGameResponse, error) {
	gameApp, gameData, err := s.validateOnboardGame(ctx, params)
	if err != nil {
		return nil, err
	}

	gameIdentifier := fmt.Sprintf("%d", gameApp.GameId)

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	if err := s.createResource(ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: gameApp.CompanyId,
		Resource: &rbacrpc.Resource{
			ExternalId: fmt.Sprintf("%d", gameApp.GameId),
			Type:       "game",
		},
	}); err != nil {
		return nil, err
	}

	if err := s.Backend.DeleteGameApplication(ctx, &rbacrpc.DeleteGameApplicationRequest{
		Id: gameApp.Id,
	}); err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	s.sendOnboardGameEmail(ctx, gameData.Name, params, gameApp)

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: auth.GetTwitchID(ctx),
		Action:       "Game approved",
		EntityType:   "Game",
		EntityID:     gameIdentifier,
		CompanyID:    common.NewSQLNullString(gameApp.CompanyId),
	})

	return &rbacrpc.OnboardGameResponse{
		GameId:   gameApp.GameId,
		GameName: gameData.Name,
	}, nil
}

func (s *Server) getGameByGameID(ctx context.Context, gameID int32) (*dclient.LocalizedGame, error) {
	gameKey := fmt.Sprintf("%d", gameID)

	// Get game from discovery
	var gamesData map[string]*dclient.LocalizedGame
	gamesData, err := s.Discovery.GetAll(ctx, []string{gameKey})
	if err != nil {
		return nil, err
	}

	gameData, ok := gamesData[gameKey]
	if !ok || gameData == nil {
		return nil, twirp.NotFoundError("game not found")
	}

	return gameData, err
}

func (s *Server) validateOnboardGame(ctx context.Context, params *rbacrpc.OnboardGameRequest) (*rbacrpc.GameApplication, *dclient.LocalizedGame, error) {
	gameApp, err := s.Backend.SelectGameApplication(ctx, params.GetGameApplicationId())
	if errorutil.IsErrNoRows(err) {
		return nil, nil, twirp.InvalidArgumentError("id", "game application not found")
	}
	if err != nil {
		return nil, nil, err
	}

	gameData, err := s.getGameByGameID(ctx, gameApp.GameId)
	if err != nil {
		return gameApp, nil, err
	}

	return gameApp, gameData, nil
}

func (s *Server) sendOnboardGameEmail(ctx context.Context, gameName string, params *rbacrpc.OnboardGameRequest, gameApp *rbacrpc.GameApplication) {
	async(ctx, func(ctx context.Context) error {
		// Get user to send the email
		owner, err := s.getCompanyOwner(ctx, gameApp.CompanyId)
		if err != nil {
			return errx.Wrap(err, "Failed to get user to send game approval email to.")
		}

		emailTraceId, emailError := s.Dart.SendDeveloperCompanyGameActionApproved(ctx, owner.TwitchID, gameName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   owner.TwitchID,
			EntityTwitchID: gameApp.CompanyId,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyGameActionApproved, emailTraceId),
			CompanyID:      gameApp.CompanyId,
		})

		return emailError
	}, logx.Fields{
		"gameApplicationID": params.GameApplicationId,
		"companyID":         gameApp.CompanyId,
		"gameName":          gameName,
	})
}
