package rbacrpcserver

import (
	"sort"
	"strings"

	"code.justin.tv/devrel/devsite-rbac/common"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"

	"github.com/twitchtv/twirp"
)

const (
	ownerRole          = common.OwnerRole
	adminRole          = common.AdminRole
	billingManagerRole = common.BillingManagerRole // with underscore to match GraphQL enum BILLING_MANAGER
	managerRole        = common.ManagerRole
	marketerRole       = common.MarketerRole
	developerRole      = common.DeveloperRole
	shadowAccountRole  = "Shadow_Account" // the role can only be created with a new user. The assignment to an existing user will be forbidden.
)

var (
	validRoles        = []string{ownerRole, adminRole, developerRole, marketerRole, managerRole, billingManagerRole}
	validRolesByName  map[string]string
	roleValidationMsg string
)

func init() {
	validRolesByName = make(map[string]string)
	for _, role := range validRoles {
		validRolesByName[strings.ToLower(role)] = role
	}

	roleValidationMsg = strings.Join(validRoles, ",")

	// shadow account has same permissions as developer
	extendPermissions(shadowAccountRole, developerRole)
	// marketer has all developer permissions
	extendPermissions(marketerRole, developerRole)
	// manager has all marketer permissions
	extendPermissions(managerRole, marketerRole)
	// billing manager has all manager permissions
	extendPermissions(billingManagerRole, managerRole)
	// admin has all manager permissions
	extendPermissions(adminRole, managerRole)
	// owner has all admin permissions
	extendPermissions(ownerRole, adminRole)

	for role, permissions := range permissionsByRole {
		// sort permissions so we can find them quickly with sort.Search
		sort.Strings(permissions)
		permissionsByRole[role] = permissions
	}
}

func SanitizeRole(field, role string) (string, error) {
	formatted, ok := validRolesByName[strings.ToLower(role)]
	if !ok {
		return "", twirp.InvalidArgumentError(field, "must be one of "+roleValidationMsg)
	}

	return formatted, nil
}

var permissionsByRole = map[string][]string{
	ownerRole: []string{
		permissions.SetBillingManager,
		permissions.AddUserBillingManager,
		permissions.ViewPaymentOnboarding,
	},
	adminRole: []string{
		permissions.EditCompanyInfo,
		permissions.RemoveUser,
	},
	billingManagerRole: []string{
		permissions.RemoveUser,
	},
	managerRole: []string{
		permissions.AddUser,
		permissions.AssignPermissions,
	},
	marketerRole: []string{
		permissions.AddGames,
		permissions.GameBoxArtEdit,
	},
	shadowAccountRole: []string{},
	developerRole: []string{
		permissions.ShowCompanyMembers,
		permissions.ViewDrops,
		permissions.CreateDrops,
		permissions.ManageDrops,
		permissions.DeleteDrops,
		permissions.GameAnalyticsView,
		permissions.InsightsOnePagerView,
		permissions.CreateExtensions,
		permissions.ManageExtensions,
		permissions.MonetizeExtensions,
		permissions.ViewExtensionInsights,
		permissions.ViewGame,
		permissions.DropsAnalyticsView,
	},
}

// extendPermissions applies permissions fromRole toRole.
func extendPermissions(toRole, fromRole string) {
	permissionsByRole[toRole] = append(permissionsByRole[toRole], permissionsByRole[fromRole]...)
}

func doesRoleHavePermission(role, permission string) bool {
	permissions, ok := permissionsByRole[role]
	if !ok {
		return false
	}

	i := sort.SearchStrings(permissions, permission)
	return i < len(permissions) && permissions[i] == permission
}
