package rbacrpcserver

import (
	"context"
	"strconv"

	"github.com/twitchtv/twirp"

	owl "code.justin.tv/web/owl/client"

	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) TransferExtensionToCompany(ctx context.Context, params *rbacrpc.TransferExtensionRequest) (*rbacrpc.Empty, error) {
	requestingTwitchID := auth.GetTwitchID(ctx)
	if requestingTwitchID == "" {
		return nil, twirp.NewError(twirp.PermissionDenied, "OAuth token required")
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"extension_id", params.ExtensionId},
	}); err != nil {
		return nil, err
	}
	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return nil, err
	}

	// DB transaction
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Check if company exists
	_, err := s.Backend.SelectCompany(ctx, params.CompanyId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.InvalidArgumentError("company_id", "is not a valid company id")
	}
	if err != nil {
		return nil, err
	}

	// Add extension to RBAC as company resource
	if err := s.createResource(ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: params.CompanyId,
		Resource: &rbacrpc.Resource{
			ExternalId: params.ExtensionId,
			Type:       permissions.Extension,
		},
	}); err != nil {
		return nil, err
	}

	// Load previous Owl client values to allow rollback
	prevClient, err := s.Owl.GetClient(ctx, params.ExtensionId)
	if errorutil.Is(err, owl.ErrInvalidClientID) {
		return nil, twirp.InvalidArgumentError("extension_id", "is not a valid client id")
	}
	if err != nil {
		return nil, err
	}
	if prevClient.OwnerID == nil {
		return nil, twirp.NewError(twirp.DataLoss, "Extension client doesn't have an owner_id")
	}

	// Detect if the previous owner of the extension is part of the same org
	prevOwnerID := strconv.Itoa(*prevClient.OwnerID)
	_, err = s.Memberships.GetMembership(ctx, params.CompanyId, prevOwnerID)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("user is not a member of the company")
	}
	if err != nil {
		return nil, err
	}

	// Update extension client in Owl
	err = s.Owl.UpdateClientOwner(ctx, owlcli.UpdateClientRequest{
		ClientID:           params.ExtensionId,
		OwnerID:            prevOwnerID,
		GroupID:            params.CompanyId,
		RequestingTwitchID: requestingTwitchID,
	})
	if err != nil {
		// errors owl.ErrInvalidClientID or owl.ErrInvalidUserID should not happen because params were already validated
		return nil, err // internal error
	}

	// DB Transaction Commit
	if err := s.Backend.Commit(ctx); err != nil {
		s.rollbackOwlClient(ctx, prevClient, requestingTwitchID)
		return nil, err
	}

	// If the request has a billing_manager_id, then the extension is assumed to be monetized.
	// Assign billing manager through another endpoint to simplify/reuse code.
	// This will have a new transaction, if it fails we need to fix manually,
	// but that is fine because this is called from Vienna only, and not too often.
	if params.BillingManagerId != "" {
		// Assign the billing manager to the extension, after it was created as a resource.
		_, err := s.SetExtensionBillingManager(ctx, &rbacrpc.SetExtensionBillingManagerRequest{
			ExtensionClientId:      params.ExtensionId,
			BillingManagerTwitchId: params.BillingManagerId,
		})
		if err != nil {
			return nil, err
		}
	}

	return &rbacrpc.Empty{}, nil
}
