package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

type InviteAction int

const (
	InviteAccept  InviteAction = 0
	InviteDecline InviteAction = 1
)

func (e InviteAction) String() string {
	actions :=
		[...]string{
			"accepted",
			"declined",
		}

	return actions[e]
}

func (s *Server) AcceptCompanyInvite(ctx context.Context, params *rbacrpc.AcceptCompanyInviteRequest) (*rbacrpc.Empty, error) {
	if err := validateAcceptCompanyInviteParams(params); err != nil {
		return nil, err
	}

	invite, err := validateInvite(ctx, s, params.CompanyInviteId, params.RequestingTwitchId, InviteAccept)
	if err != nil {
		return nil, err
	}

	isValidTwitchInvitee, errUser := s.Users.IsValidTwitchID(ctx, invite.InviteeTwitchId)
	if errUser != nil {
		return nil, errUser
	}
	if !isValidTwitchInvitee {
		return nil, twirp.NewError(twirp.PermissionDenied, "Invitee does not exists")
	}

	memb := memberships.Membership{
		CompanyID: invite.CompanyId,
		TwitchID:  invite.InviteeTwitchId,
		Role:      invite.Role,
		FirstName: params.FirstName,
		LastName:  params.LastName,
		DevTitle:  params.Title,
		DevEmail:  "", // TODO: remove this column after Multi-Org (DXL-583) released
	}

	// Ensure user doesn't already have invite.
	// Ensure the user is not already a member in the same org (prevent duplicates) or another org (prevent multi-org)
	inviteValidator, inviteValidatorError := s.IsUserValidForCompanyInvite(ctx, &rbacrpc.ValidUserCompanyInviteRequest{
		TwitchId:  invite.InviteeTwitchId,
		CompanyId: invite.CompanyId,
	})

	if inviteValidatorError != nil {
		return nil, err
	}

	// Ensure billing_manager is possible if assigned
	if memb.Role == billingManagerRole {
		if err := s.validateCanBeBillingManager(ctx, memb.TwitchID); err != nil {
			return nil, err
		}
	}

	// TODO[Multi-Org]: Only check if they exist in the invited org.
	// Ensure the user is not already a member in the same org (prevent duplicates) or another org (prevent multi-org)
	if !inviteValidator.NotExistingMember {
		//Delete Company Invite row
		if err := s.Backend.DeleteCompanyInvite(ctx, invite.Id); err != nil {
			return nil, err
		}
		return nil, twirp.NewError(twirp.AlreadyExists, "Membership already exists. Invite deleted")
	}

	if !inviteValidator.WithinMembershipLimit {
		return nil, twirp.NewError(twirp.OutOfRange, "User is already at max memberships. ")
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Create Membership
	if err := s.Memberships.InsertMembership(ctx, &memb); err != nil {
		return nil, err
	}

	//Delete Company Invite row
	if err := s.Backend.DeleteCompanyInvite(ctx, invite.Id); err != nil {
		return nil, err
	}

	//Add Action History
	s.auditUserInviteChange(ctx, UserInviteAction{
		CurrentUserTwitchID: params.RequestingTwitchId, // twitchID of the main actor of the update (inviter)
		EntityTwitchID:      memb.TwitchID,             // twitchID of the person being invited (invitee)
		ActionFormat:        fmt.Sprintf("Edit: Invite accepted for role of %s", memb.Role),
		CompanyID:           memb.CompanyID, // Company/organization the user is being invited to join
	})

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	return &rbacrpc.Empty{}, nil
}

func (s *Server) DeclineCompanyInvite(ctx context.Context, params *rbacrpc.DeclineCompanyInviteRequest) (*rbacrpc.Empty, error) {

	invite, err := validateInvite(ctx, s, params.CompanyInviteId, params.RequestingTwitchId, InviteDecline)
	if err != nil {
		return nil, err
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	//Delete invite
	if err := s.Backend.DeleteCompanyInvite(ctx, invite.Id); err != nil {
		return nil, err
	}

	actionMsg := "Edit: Invite declined for role of %s"

	if params.RequestingTwitchId != invite.InviteeTwitchId {
		actionMsg = "Delete: Invite deleted for role %s"
	}

	//Add Action History
	s.auditUserInviteChange(ctx, UserInviteAction{
		CurrentUserTwitchID: params.RequestingTwitchId, // twitchID of the main actor of the update (inviter)
		EntityTwitchID:      invite.InviteeTwitchId,    // twitchID of the person being invited (invitee)
		ActionFormat:        fmt.Sprintf(actionMsg, invite.Role),
		CompanyID:           invite.CompanyId, // Company/organization the user is being invited to join
	})

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	return &rbacrpc.Empty{}, nil
}

func validateAcceptCompanyInviteParams(params *rbacrpc.AcceptCompanyInviteRequest) error {
	return errorutil.ValidateRequiredArgs(errorutil.Args{
		{"first name", params.FirstName},
		{"last name", params.LastName},
		{"title", params.Title},
		{"invite ID", params.CompanyInviteId},
		{"requesting twitch ID", params.RequestingTwitchId},
	})
}

func validateInvite(ctx context.Context, s *Server, inviteId string, requestingTwitchId string, action InviteAction) (*rbacrpc.CompanyInvite, error) {
	//fetch invite
	invite, err := s.Backend.SelectCompanyInvite(ctx, inviteId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("company invite not found")
	}
	if err != nil {
		return nil, err // internal error
	}

	//Validate requesting twitch id is id on invite or admin account
	if !auth.IsWhitelistAdmin(ctx) {
		if action == InviteDecline && invite.InviteeTwitchId != requestingTwitchId {

			canDelete := true
			// Get requesting user role in company
			requestingMemb, err := s.Memberships.GetMembership(ctx, invite.CompanyId, requestingTwitchId)
			if errorutil.IsErrNoRows(err) {
				canDelete = false
			}
			if err != nil {
				return nil, err
			}

			if !doesRoleHavePermission(requestingMemb.Role, permissions.AddUser) {
				canDelete = false
			}

			if !canDelete {
				return nil, twirp.NewError(twirp.PermissionDenied, "user cannot delete company invite")
			}

		} else if invite.InviteeTwitchId != requestingTwitchId {
			return nil, twirp.NewError(twirp.PermissionDenied, fmt.Sprintf("user cannot %s company invite", action.String()))
		}

	}

	return invite, nil
}
