package rbacrpcserver

import (
	"context"

	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *Server) UpdateUser(ctx context.Context, params *rbacrpc.UpdateUserRequest) (*rbacrpc.Empty, error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"twitch_id", params.TwitchId},
		{"first_name", params.FirstName},
		{"last_name", params.LastName},
		{"email", params.DevEmail},
		{"title", params.DevTitle},
	}); err != nil {
		return nil, err
	}
	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return nil, err
	}

	// Ensure permissions: for now this endpoint can only be called from Vienna
	if !auth.IsWhitelistAdmin(ctx) {
		return nil, twirp.NewError(twirp.PermissionDenied, "cannot update membership")
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Load membership
	memb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.TwitchId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("membership with company_id and twitch_id not found")
	}
	if err != nil {
		return nil, err
	}

	// Update membership
	memb.FirstName = params.FirstName
	memb.LastName = params.LastName
	memb.DevEmail = params.DevEmail
	memb.DevTitle = params.DevTitle
	err = s.Memberships.UpdateMembership(ctx, &memb)
	if err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	return &rbacrpc.Empty{}, nil
}
