package rbacrpcserver

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"github.com/stretchr/testify/require"
)

func TestCheckWithinUserMembershipLimit(t *testing.T) {
	server, _ := NewTestServer()

	ctx := context.Background()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	usersFake := &users.NoopClient{}
	server.Users = usersFake

	twitchId := "123"
	maxLimit := memberships.MaxAllowedMemberships
	maxStaffLimit := memberships.MaxStaffMemberships

	if maxLimit != 0 {
		t.Run("Passes Membership Limit", func(t *testing.T) {
			fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, int32(maxLimit-1), nil)
			usersFake.GetStaffUser = false
			resp, err := server.CheckWithinUserMembershipLimit(ctx, twitchId)

			require.NoError(t, err)
			require.True(t, resp)

		})

		t.Run("Passes Staff Membership Limit", func(t *testing.T) {
			fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, int32(maxStaffLimit-1), nil)
			usersFake.GetStaffUser = true
			resp, err := server.CheckWithinUserMembershipLimit(ctx, twitchId)

			require.NoError(t, err)
			require.True(t, resp)

		})

		t.Run("Fails Membership Limit", func(t *testing.T) {
			fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, int32(maxLimit), nil)
			usersFake.GetStaffUser = false
			resp, err := server.CheckWithinUserMembershipLimit(ctx, twitchId)

			require.NoError(t, err)
			require.False(t, resp)

		})

		t.Run("Internal Error", func(t *testing.T) {
			fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, errors.New("xxx"))
			usersFake.GetStaffUser = false
			resp, err := server.CheckWithinUserMembershipLimit(ctx, twitchId)

			require.Error(t, err)
			require.False(t, resp)

		})
	} else {
		t.Run("Membership Limit check turned off", func(t *testing.T) {
			fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, int32(maxLimit-1), nil)
			usersFake.GetStaffUser = false
			resp, err := server.CheckWithinUserMembershipLimit(ctx, twitchId)

			require.NoError(t, err)
			require.False(t, resp)

		})
	}
}
