package rbacrpcserver

import (
	"context"
	"database/sql"
	"errors"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/passport/passportfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"
)

func TestIsUserValidForCompanyInvite_Success(t *testing.T) {
	server, rbacBackend := NewTestServer()

	ctx := context.Background()
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	request := validUserCompanyInviteRequestTemp()

	rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(false, nil)
	fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
	fakeMemberships.GetMembershipReturns(memberships.Membership{}, sql.ErrNoRows)
	fakePassport.GetTwoFactorEnabledReturns(true, nil)

	resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

	require.NoError(t, err)
	require.NotNil(t, resp)

	require.True(t, resp.VerifiedEmail)
	require.True(t, resp.NoInvites)
	require.True(t, resp.WithinMembershipLimit)
	require.True(t, resp.NotExistingMember)
	require.True(t, resp.Has_2Fa)

}

func TestIsUserValidForCompanyInvite_InvalidUser(t *testing.T) {
	server, _ := NewTestServer()

	ctx := context.Background()
	usersFake := &users.NoopClient{}
	server.Users = usersFake

	request := validUserCompanyInviteRequestTemp()

	t.Run("User Not Found", func(t *testing.T) {
		usersFake.GetUserByIDReturnsError = errors.New("User not found")
		resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

		require.Error(t, err)
		require.EqualError(t, err, "twirp error not_found: user does not exists")
		require.Nil(t, resp)
	})

	t.Run("Internal server error", func(t *testing.T) {
		usersFake.GetUserByIDReturnsError = errors.New("xxx")
		resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

		require.Error(t, err)
		require.Nil(t, resp)
	})
}

func TestIsUserValidForCompanyInvite_FailCase(t *testing.T) {
	server, rbacBackend := NewTestServer()

	ctx := context.Background()
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	request := validUserCompanyInviteRequestTemp()

	rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(true, nil)
	fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
	fakeMemberships.GetMembershipReturns(memberships.Membership{}, nil)
	fakePassport.GetTwoFactorEnabledReturns(false, nil)

	resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

	require.NoError(t, err)
	require.NotNil(t, resp)
	require.False(t, resp.NoInvites)
	require.False(t, resp.NotExistingMember)
	require.False(t, resp.Has_2Fa)

}

func TestIsUserValidForCompanyInvite_InternalError(t *testing.T) {
	server, rbacBackend := NewTestServer()

	ctx := context.Background()
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	request := validUserCompanyInviteRequestTemp()

	t.Run("2fa failed", func(t *testing.T) {
		rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(false, nil)
		fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
		fakeMemberships.GetMembershipReturns(memberships.Membership{}, nil)
		fakePassport.GetTwoFactorEnabledReturns(false, errors.New("xxx"))

		resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

		require.Error(t, err)
		require.Nil(t, resp)
	})

	t.Run("Membership failed", func(t *testing.T) {
		rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(false, nil)
		fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
		fakeMemberships.GetMembershipReturns(memberships.Membership{}, errors.New("xxx"))

		resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

		require.Error(t, err)
		require.Nil(t, resp)
	})

	t.Run("Company Invites failed", func(t *testing.T) {
		rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(false, errors.New("xxx"))

		resp, err := server.IsUserValidForCompanyInvite(ctx, &request)

		require.Error(t, err)
		require.Nil(t, resp)
	})

}

func validUserCompanyInviteRequestTemp() rbacrpc.ValidUserCompanyInviteRequest {
	return rbacrpc.ValidUserCompanyInviteRequest{
		TwitchId:  "123",
		CompanyId: common.NewUUID(),
	}
}
