package internal

import (
	"bytes"
	"fmt"
	"go/ast"
	"go/build"
	"go/format"
	"go/parser"
	"go/token"
	"io"
	"io/ioutil"
	"path/filepath"
	"strings"

	"github.com/pkg/errors"
	"github.com/sqs/goreturns/returns"
	"golang.org/x/tools/imports"
)

var (
	ErrNoSuchInterface = errors.New("no such interface")
)

type Config struct {
	Dir  string
	Name string // target interface to implement

	WithTimings bool // true if flag --timings is passed. Adds TimingFunc field to the errxer struct.
}

type printConfig struct {
	Interface string
	Directory string

	WithTimings bool

	ExternalInterface bool
	ExternalDirectory string
}

func Filename(name string) string {
	name = strings.Replace(name, ".", "_", 1)
	return strings.ToLower(name) + "_errxer.gen.go"
}

func Run(cf Config) error {
	buf := &bytes.Buffer{}

	fileIdentifier := cf.Name

	pc := printConfig{
		Directory:   cf.Dir,
		Interface:   cf.Name,
		WithTimings: cf.WithTimings,
	}

	pathPieces := strings.Split(cf.Name, "/")
	if len(pathPieces) > 1 {
		iPieces := strings.Split(pathPieces[len(pathPieces)-1], ".")
		if len(iPieces) == 2 {
			pc.ExternalInterface = true
			pc.Interface = iPieces[1]
			path := pathPieces[0 : len(pathPieces)-1]
			path = append(path, iPieces[0])
			pc.ExternalDirectory = strings.Join(path, "/")

			fileIdentifier = strings.Join(iPieces, ".")
		} else {
			return errors.New("expected interface to be in format package.name")
		}
	}

	if err := Print(buf, pc); err != nil {
		return errors.Wrap(err, "failed to generate code")
	}

	imported, err := imports.Process("", buf.Bytes(), nil)
	if err != nil {
		fmt.Println(buf.String())
		return errors.Wrap(err, "failed to imports")
	}

	returned, err := returns.Process("", "", imported, &returns.Options{
		Fragment: true,
	})
	if err != nil {
		return errors.Wrap(err, "failed to goreturns")
	}

	formatted, err := format.Source(returned)
	if err != nil {
		return errors.Wrap(err, "failed to format code")
	}

	return ioutil.WriteFile(Filename(fileIdentifier), formatted, 0644)
}

func Print(w io.Writer, c printConfig) error {
	// Parse files in the given directory
	fset := token.NewFileSet()

	if c.ExternalInterface {
		pkg, err := build.Default.Import(c.ExternalDirectory, c.Directory, build.AllowBinary)
		if err != nil {
			return fmt.Errorf("unable to find interface %s: %s", c.Interface, err)
		}
		c.Directory = pkg.Dir
	}

	packagesByName, err := parser.ParseDir(fset, c.Directory, nil, 0)
	if err != nil {
		return err
	}

	// Find interfaces and the target interface
	interfaces := FindInterfaces(packagesByName)
	targetInterface, ok := interfaces[c.Interface]
	if !ok {
		return ErrNoSuchInterface
	}

	// Start with the generated file comment
	if _, err := io.WriteString(w, "// Code generated by errxer. DO NOT EDIT.\n"); err != nil {
		return err
	}

	// Write package name to output file
	packageName := filepath.Base(c.Directory)
	_, err = io.WriteString(w, fmt.Sprintf("package %s\n", packageName))
	if err != nil {
		return err
	}

	// Define struct data with the interface definition
	var sd structData
	sd.Type = c.Interface
	sd.WithTimings = c.WithTimings
	if c.ExternalInterface {
		sd.Package = filepath.Base(c.ExternalDirectory)

		targetInterface.FileImports = append(targetInterface.FileImports, importData{
			Path: fmt.Sprintf("%q", c.ExternalDirectory),
		})
	}
	sd.Imports = targetInterface.FileImports

	// Methods
	err = sd.fillFieldList(targetInterface.Methods, interfaces)
	if err != nil {
		return err
	}

	// Render template
	_, err = sd.Write(w)
	return err
}

func ImportsDataList(imports []*ast.ImportSpec) []importData {
	importsData := make([]importData, len(imports))
	for i, importSpec := range imports {
		nickname := ""
		if importSpec.Name != nil {
			nickname = importSpec.Name.Name
		}

		importsData[i] = importData{
			Nickname: nickname,
			Path:     importSpec.Path.Value,
		}
	}
	return importsData
}

// Find all the interfaces on the packages and return a map by interface type name.
func FindInterfaces(packagesByName map[string]*ast.Package) map[string]*interfaceData {
	interfaces := map[string]*interfaceData{}
	for _, packageAST := range packagesByName {
		for _, fileAST := range packageAST.Files {
			// TODO: Capture the imports here and insert them into generated file
			// so that types match correctly
			for _, decl := range fileAST.Decls {
				genDecl, ok := decl.(*ast.GenDecl)
				if !ok {
					continue
				}
				if genDecl.Tok != token.TYPE {
					continue // not a token to be concerned about
				}

				for _, spec := range genDecl.Specs {
					if typeSpec, ok := spec.(*ast.TypeSpec); ok {
						if interfaceType, ok := typeSpec.Type.(*ast.InterfaceType); ok {
							interfaces[typeSpec.Name.Name] = &interfaceData{
								TypeSpec:    typeSpec,
								Name:        typeSpec.Name.Name,
								Methods:     interfaceType.Methods,
								FileImports: ImportsDataList(fileAST.Imports),
							}
						}
					}
				}
			}
		}
	}
	return interfaces
}
