package passport

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/identity/passport/passport/errors"
)

const (
	defaultTimingXactName = "passport"
	authorizationHeader   = "Authorization"
	LoginAvailableStatus  = http.StatusNoContent
)

type Client interface {
	GetTwoFactorEnabled(ctx context.Context, uid string, reqOpts *twitchclient.ReqOpts) (bool, error)
	VerifySudoToken(ctx context.Context, uid string, cookie string, reqOpts *twitchclient.ReqOpts) error
	GetUsernameAvailable(ctx context.Context, username string, reqOpts *twitchclient.ReqOpts) (bool, error)
}

type clientImpl struct {
	twitchclient.Client
}

type checkTwoFactorResponse struct {
	AuthyID int `json:"authy_id"`
}

type UserIDInput struct {
	UserID string `json:"user_id"`
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &clientImpl{twitchClient}, err
}

func (c *clientImpl) GetTwoFactorEnabled(ctx context.Context, uid string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	if uid == "" {
		return false, errors.MissingUserID
	}

	combinedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName: "service.passport.check_two_factor",
	})

	req, err := c.NewRequest("GET", fmt.Sprintf("/users/%s/2fa", uid), nil)
	if err != nil {
		return false, err
	}

	var checkTwoFactorResponse *checkTwoFactorResponse
	resp, err := c.DoJSON(ctx, &checkTwoFactorResponse, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			return false, nil
		}
		return false, err
	}

	return true, nil
}

func (c *clientImpl) GetUsernameAvailable(ctx context.Context, username string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	if username == "" {
		return false, errors.MissingUsername
	}

	combinedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName: "service.passport.check_username_available",
	})

	req, err := c.NewRequest("GET", fmt.Sprintf("/usernames/%s?trusted_request=true", username), nil)
	if err != nil {
		return false, err
	}

	resp, err := c.DoNoContent(ctx, req, combinedOpts)

	if err == nil {
		return resp.StatusCode == LoginAvailableStatus, nil
	}

	return false, err
}

type VerifySudoTokenInput struct {
	SudoToken string `json:"sudo_token"`
	UserID    string `json:"user_id"`
}

func (c *clientImpl) VerifySudoToken(ctx context.Context, uid string, cookie string, reqOpts *twitchclient.ReqOpts) error {
	if cookie == "" {
		return errors.MissingSudoToken
	}
	if uid == "" {
		return errors.MissingUserID
	}

	combinedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName: "service.passport.verify_sudo_token",
	})

	jsonBytes, err := json.Marshal(&VerifySudoTokenInput{
		SudoToken: cookie,
		UserID:    uid,
	})
	if err != nil {
		return errors.InvalidJSON
	}

	req, err := c.NewRequest(http.MethodPost, "/validate/sudo", bytes.NewBuffer(jsonBytes))
	if err != nil {
		return err
	}

	resp, err := c.DoNoContent(ctx, req, combinedOpts)
	if err != nil {
		e := errors.InvalidSudoToken
		if resp != nil && resp.StatusCode == http.StatusBadRequest {
			e = errors.InvalidJSON
		}
		return e
	}

	return nil
}
