package logger

import (
	"errors"
	"io"
	"log"
	"net/http"
	"os"

	"code.justin.tv/identity/passport/clients/rollbar"

	"golang.org/x/net/context"
)

// LstdFlags sets timestamp in seconds: <msg>
var LstdFlags = log.LstdFlags

// DefaultLogger logs to stdout with LstdFlags with Severity 0.
var DefaultLogger Logger = &logger{
	logger:    log.New(os.Stdout, "", LstdFlags),
	verbosity: severities[Debug],
	rollbar:   rollbar.NewLogger("", "development"),
}

// Severity tells the Logger what level to log.
// Messages logged at a severity greater than the configured Severity
// will be logged.
type Severity uint8

// Severities range from Debug (0) to Fatal (6).
const (
	Debug Severity = iota
	Trace
	Info
	Warn
	Err
	Critical
	Fatal
)

type severity struct {
	priority    uint8
	name        string
	rollbarName string
}

var severities = []severity{
	severity{
		priority:    0,
		name:        "DEBUG",
		rollbarName: "debug",
	},
	severity{
		priority:    1,
		name:        "TRACE",
		rollbarName: "debug",
	},
	severity{
		priority:    2,
		name:        "INFO",
		rollbarName: "info",
	},
	severity{
		priority:    3,
		name:        "WARN",
		rollbarName: "warning",
	},
	severity{
		priority:    4,
		name:        "ERR",
		rollbarName: "error",
	},
	severity{
		priority:    5,
		name:        "CRITICAL",
		rollbarName: "critical",
	},
	severity{
		priority:    6,
		name:        "FATAL",
		rollbarName: "critical",
	},
}

// NewLogger creates a new Logger.
// eg DefaultLogger = NewLogger(file, logger.Warn, logger.LStdFlags, "asdfas")
func NewLogger(w io.Writer, verbosity Severity, flags int, rollbarToken string) Logger {
	env := os.Getenv("DEPLOY_ENV")
	if env == "" {
		env = "development"
		w = io.MultiWriter(w, os.Stdout)
	}

	return &logger{
		logger:    log.New(w, "", flags),
		verbosity: severities[verbosity],
		rollbar:   rollbar.NewLogger(rollbarToken, env),
	}
}

// Logger specifies a struct that can write anything from debug statements to fatal errors to a destination of its
// choice, with the option to include contextual information from a relevant HTTP request.
type Logger interface {
	Log(context.Context, Severity, string)
}

type logger struct {
	logger    *log.Logger
	verbosity severity
	rollbar   rollbar.ErrorLogger
}

// Log prints a string to stdout. If the severity is Warn or above, it also reports the string to Rollbar.
func (l *logger) Log(ctx context.Context, s Severity, m string) {
	if severities[s].priority >= severities[Warn].priority {
		l.rollbar.Log(ctx, severities[s].rollbarName, errors.New(m))
	}
	l.stdout(s, m)
}

func (l *logger) stdout(s Severity, m string) {
	if severities[s].priority >= l.verbosity.priority {
		if s == Fatal {
			l.logger.Fatalf("%s: %s\n", severities[s].name, m)
		} else {
			l.logger.Printf("%s: %s\n", severities[s].name, m)
		}
	}
}

// Log writes to the DefaultLogger.
func Log(ctx context.Context, s Severity, m string) {
	DefaultLogger.Log(ctx, s, m)
}

// LogRequest writes to the DefaultLogger.
func LogRequest(ctx context.Context, s Severity, r *http.Request, m string) {
	DefaultLogger.Log(ctx, s, m)
}
