package errors

import (
	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

var (
	CAPTCHA = &WrappedError{
		InternalError: "captcha incorrect",
		ExternalMessage: i18n.T(
			"Please complete the CAPTCHA correctly.",
			"Error text displayed when a user completes the CAPTCHA incorrectly or does not complete the CAPTCHA.",
		),
		Severity: logger.Trace,
		Code:     1000,
	}

	GenerateRedirect = &WrappedError{
		InternalError:   "couldn't generate redirect URI",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Err,
		Code:            1001,
	}

	InvalidJSON = &WrappedError{
		InternalError:   "failed to decode JSON",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            1002,
	}

	NoClientID = &WrappedError{
		InternalError: "missing client_id",
		ExternalMessage: i18n.T(
			"Missing client_id",
			"Error text displayed when a client_id is missing.",
		),
		Severity: defaultLogLevel,
		Code:     1003,
	}

	NotLoggedIn = &WrappedError{
		InternalError: "user not logged in",
		ExternalMessage: i18n.T(
			"You must be logged in to do that.",
			"Error message when the user is not logged in",
		),
		Severity: logger.Trace,
		Code:     1004,
	}

	// NoTwoFactor is an error returned when a lookup on a given user's two-factor information
	// yields no result (404)
	NoTwoFactor = &WrappedError{
		InternalError: "no second factor found",
		ExternalMessage: i18n.T(
			"No second-factor found",
			"Error text displayed when a user tries to interact with two-factor auth without a second factor set",
		),
		Severity: logger.Trace,
		Code:     1005,
	}

	RenderTemplate = &WrappedError{
		InternalError:   "failure to render template",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Fatal,
		Code:            1006,
	}

	/// RedirectMismatch is a OAuth Client error when a request's redirect_uri doesn't
	// match the redirect URI registered for a given Client
	RedirectMismatch = &WrappedError{
		InternalError: "redirect mismatch on authorization request",
		Severity:      defaultLogLevel,
		Code:          1007,
	}

	MissingSudoToken = &WrappedError{
		InternalError: "missing sudo token",
		Severity:      logger.Trace,
		Code:          1008,
	}

	InvalidSudoToken = &WrappedError{
		InternalError: "invalid sudo token",
		Severity:      logger.Trace,
		Code:          1009,
	}

	// IncrAttempts represents a failure to increment throttler attempts.
	IncrAttempts = &WrappedError{
		InternalError:   "unable to increment attempts",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            1010,
	}

	LastTimeout = &WrappedError{
		InternalError:   "unable to get last timeout",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            1011,
	}

	SetTimeout = &WrappedError{
		InternalError:   "unable to set new timeout",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            1012,
	}

	UsersServiceUnexpected = &WrappedError{
		InternalError:   "unexpected error with Users service",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            1013,
	}

	UserNotFound = &WrappedError{
		InternalError:   "user does not exist",
		ExternalMessage: IncorrectCredentials.ExternalMessage,
		Severity:        logger.Trace,
		Code:            1014,
	}

	// InvalidCaptchaProof indicates that the captcha proof was wrong.
	InvalidCaptchaProof = &WrappedError{
		InternalError: "invalid captcha proof",
		ExternalMessage: i18n.T(
			"Invalid captcha proof. Please try logging in again",
			"Login error",
		),
		Severity: defaultLogLevel,
		Code:     5018,
	}

	MissingUserID = &WrappedError{
		InternalError: "missing user_id",
		ExternalMessage: i18n.T(
			"User ID missing from request",
			"Error text displayed when we are un able to verify the two factor token with authy.",
		),
		Severity: defaultLogLevel,
		Code:     5019,
	}

	UnsupportedFilename = &WrappedError{
		InternalError: "unsupported filename",
		ExternalMessage: i18n.T(
			"Unsupported filename",
			"Error text displayed when we are given an invalid filename.",
		),
		Severity: defaultLogLevel,
		Code:     5020,
	}
)
