package i18n

import (
	"crypto/md5"
	"fmt"
	"io"
	"log"
	"os"
	"path/filepath"
	"strconv"

	"code.justin.tv/web/go-i18n/smartling"
)

var (
	SupportedLangsToCanonicalLabels = map[string]string{
		"en-US": "en",
		"ar-SA": "ar",
		"bg-BG": "bg",
		"cs-CZ": "cs",
		"da-DK": "da",
		"de-DE": "de",
		"el-GR": "el",
		"es-ES": "es",
		"es-MX": "es-mx",
		"fi-FI": "fi",
		"fr-FR": "fr",
		"hu-HU": "hu",
		"it-IT": "it",
		"ja-JP": "ja",
		"ko-KR": "ko",
		"nl-NL": "nl",
		"no-NO": "no",
		"pl-PL": "pl",
		"pt-BR": "pt-br",
		"pt-PT": "pt",
		"ro-RO": "ro",
		"ru-RU": "ru",
		"sk-SK": "sk",
		"sv-SE": "sv",
		"th-TH": "th",
		"tr-TR": "tr",
		"vi-VN": "vi",
		"zh-CN": "zh-cn",
		"zh-TW": "zh-tw",
	}
)

func FilterProjectSupportedLangs(projectDetails *smartling.GetProjectDetailsOutput) (map[string]string, error) {
	projectSupportedLangs := map[string]string{}

	sourceLabel, ok := SupportedLangsToCanonicalLabels[projectDetails.SourceLocaleID]
	if !ok {
		return nil, fmt.Errorf("support for locale %s needs to be added", projectDetails.SourceLocaleID)
	}

	projectSupportedLangs[projectDetails.SourceLocaleID] = sourceLabel

	for _, locale := range projectDetails.TargetLocales {
		label, ok := SupportedLangsToCanonicalLabels[locale.LocaleID]
		if !ok {
			return nil, fmt.Errorf("support for locale %s needs to be added", locale.LocaleID)
		}
		projectSupportedLangs[locale.LocaleID] = label
	}

	return projectSupportedLangs, nil
}

// String represents an English phrase to be internationalized.
type String struct {
	Phrase   string
	Context  string
	Position *StringPosition
	FileType string
	hash     StringHash
}

type StringPosition struct {
	FileName string // relative to $GOPATH/src/
	Line     int
}

func NewStringPosition(fpath string, line int) (*StringPosition, error) {
	relpath, err := filepath.Rel(os.Getenv("GOPATH")+"/src", filepath.Clean(fpath))
	if err != nil {
		return nil, err
	}

	return &StringPosition{
		FileName: relpath,
		Line:     line,
	}, nil
}
func (s *StringPosition) String() string {
	return s.FileName + ":" + strconv.Itoa(s.Line)
}

// StringHash is a typedef to represent an i18n.String's unique Hash
type StringHash string

// Hash returns a unique hash of an i18n.String
func (s *String) Hash() StringHash {
	if s.hash != "" {
		return s.hash
	}
	h := md5.New()
	if _, err := io.WriteString(h, s.Phrase); err != nil {
		log.Printf("warning: failed to write phrase to md5: %v", err)
	}
	if _, err := io.WriteString(h, s.Context); err != nil {
		log.Printf("warning: failed to write contxt to md5: %v", err)
	}
	s.hash = StringHash(fmt.Sprintf("%x", h.Sum(nil)))
	return s.hash
}

type Translator interface {
	Translate(s String, lang string) string
}

var babel Translator

func SetTranslator(t Translator) Translator {
	babel = t
	return babel
}

func (s *String) Translate(lang string) string {
	return babel.Translate(*s, lang)
}

// T returns an String interface. The parser will look for calls to this
// function when generating translations.json
func T(phrase, context string) *String {
	s := &String{
		Phrase:  phrase,
		Context: context,
	}
	return s
}

// Implement sort.Interface for []String
type SortStrings []String

func (s SortStrings) Len() int      { return len(s) }
func (s SortStrings) Swap(i, j int) { s[i], s[j] = s[j], s[i] }

// Sort preference:
// 1) Context
// 4) Hash
func (s SortStrings) Less(i, j int) bool {
	si, sj := s[i], s[j]
	if si.Context != sj.Context {
		return si.Context < sj.Context
	}
	return si.Hash() < sj.Hash()
}
