import { DEFAULT_SG_ID, DISCUSS_ROLE, HOSTED_ZONE_ID, LATEST_UBUNUTU, PAGERDUTY_SNS, PREFIX_URL, PROD_ACCOUNT_ID, PROJECT_URL } from './consts';
import { StackProps, Stack, Construct, Tag, Duration, Tags } from '@aws-cdk/core';
import { SecurityGroup, IVpc, Vpc, Instance, InstanceType, MachineImage, LookupMachineImage, CloudFormationInit, InitFile, InitCommand, EbsDeviceVolumeType, Protocol, UserData, InstanceClass, InstanceSize, SubnetType, CfnEIP, CfnEIPAssociation} from '@aws-cdk/aws-ec2';
import { BlockDeviceVolume } from '@aws-cdk/aws-autoscaling'
import { ApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';
import { VpcStack } from './vpc-stack';
import { Role } from '@aws-cdk/aws-iam';
import { ARecord, HostedZone, RecordTarget } from '@aws-cdk/aws-route53';
import { Alarm, Metric} from '@aws-cdk/aws-cloudwatch'
import { SnsAction } from '@aws-cdk/aws-cloudwatch-actions'
import { Topic } from '@aws-cdk/aws-sns';

interface DiscussStackProps extends StackProps {
  vpc: IVpc;
  generalVpcESecurityGroup: SecurityGroup;
  vpcStack: VpcStack;
}

/*
 *
 * When deploying this stack you will need to deploy only this stack and then update consts.ts ECR's value
 * as well as adding its IAM Role to S2S. You can find this information in the logs of the failing to start tasks.
 *
 */
export class DiscussStack extends Stack {
  public Discuss_PUBLIC_LB: ApplicationLoadBalancer;
  constructor(scope: Construct, id: string, props: DiscussStackProps) {
    super(scope, id, props);

    // iam role
    const role = Role.fromRoleArn(this, 'discuss_empty_production', DISCUSS_ROLE, {
      // Set 'mutable' to 'false' to use the role as-is and prevent adding new
      // policies to it. The default is 'true', which means the role may be
      // modified as part of the deployment.
      mutable: false,
    });
    
    // asg & ec2 settings
    const m4_2xlarge = new InstanceType("m4.2xlarge")
    const image = new LookupMachineImage({
              name: LATEST_UBUNUTU, 
              filters: {["name"]:["ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-20210223"]}
    })
    
    const scripts = UserData.forLinux()
    scripts.addCommands(
      '#!/bin/bash -xe',
      'wget https://s3.us-west-2.amazonaws.com/amazoncloudwatch-agent-us-west-2/ubuntu/amd64/latest/amazon-cloudwatch-agent.deb',
      'sudo dpkg -i -E ./amazon-cloudwatch-agent.deb',
      'apt update -y',
      'apt upgrade -y',
      'apt install awscli -y',
      'wget -qO- https://get.docker.com/ | sh',
      'mkdir /var/discourse',
      'git clone https://github.com/discourse/discourse_docker.git /var/discourse',
      // the below scripts need to be run manually due to the way discourse works
      // cd /var/discourse/
      // ./discourse-setup
      // start wizard answers
      // hostname = (grab from route53. e.g. discuss.dev.twitch.tv) 
      // DISCOURSE_DEVELOPER_EMAILS: 'developer-advocacy@twitch.tv'
      // DISCOURSE_SMTP_ADDRESS: email-smtp.us-west-2.amazonaws.com
      // DISCOURSE_SMTP_PORT: 25
      // DISCOURSE_SMTP_USER_NAME: (get from ssm parameter store)
      // DISCOURSE_SMTP_PASSWORD: (get from ssm parameter store)
      // DISCOURSE_NOTIFICATION_EMAIL: no-reply@twitch.tv
      // Let's Encrypt warnings: 'developer-advocacy@twitch.tv'
      // end wizard answers
      // nano containers/app.yml,
      // --- 1. delete postgres.temple.yml line
      // --- 2. add DISCOURSE_DB_USERNAME: (get from secrets manager)
      // --- 3. add DISCOURSE_DB_PASSWORD: (get from secrets manager)
      // --- 4. add DISCOURSE_DB_HOST: discussdb.cluster-cpukrnzx3ybc.us-west-2.rds.amazonaws.com
      // './launcher rebuild app'
      )

    const eip = new CfnEIP(this, 'eip')

    const ec2 = new Instance(this, 'ec2', {
      vpc: props.vpc,
      role,
      securityGroup: props.generalVpcESecurityGroup,
      instanceType: m4_2xlarge,
      blockDevices: [{
        deviceName: "/dev/sda1", 
        volume: 
          BlockDeviceVolume.ebs(200, {
            deleteOnTermination: false
          })
        }],
        userData: scripts,
      machineImage: image,
      }
    );    
    Tags.of(ec2).add("Name", "Discuss DevForums")   

    const ec2EipAssoc = new CfnEIPAssociation(this, "Ec2Association", {
      eip: eip.ref,
      instanceId: ec2.instanceId
    })
    
    
    const defaultSG = SecurityGroup.fromLookup(
      this,
      'DefaultSG',
      DEFAULT_SG_ID
    );
     ec2.addSecurityGroup(defaultSG)

    const hostedZone = HostedZone.fromHostedZoneAttributes(this, "domain-zone", {
      hostedZoneId: HOSTED_ZONE_ID,
      zoneName: PROJECT_URL
    });

    // const aRecord = new ARecord(this, "discussARecord", {
    //   recordName: PREFIX_URL,
    //   zone: hostedZone,
    //   ttl: Duration.minutes(1),
    //   target: RecordTarget.fromIpAddresses(eip.ref)
    // })

    const cpuMetric = new Metric({
      namespace: 'AWS/EC2',
      metricName: 'CPUUtilization',
      statistic: 'Average',
      dimensions: {
        InstanceId: ec2.instanceId 
      }
    })
    const diskMetric = new Metric({
      namespace: 'CWAgent',
      metricName: 'disk_used_percent',
      statistic: 'Average',
      dimensions: {
        path: '/',
        InstanceId: ec2.instanceId,
        ImageId: ec2.instance.imageId,
        InstanceType: ec2.instance.instanceType,
        device: 'xvda1',
        fstype: 'ext4',
      }
    })
    
    const cpuAlarm = new Alarm(this, 'DiscussCPUAlarm', {
      metric: cpuMetric,
      threshold: 90,
      evaluationPeriods: 5,
      datapointsToAlarm: 4,
    });  
    const diskAlarm = new Alarm(this, 'DiscussCPUDiskWarning', {
      metric: diskMetric,
      threshold: 80,
      evaluationPeriods: 5,
      datapointsToAlarm: 4,
    });    

    cpuAlarm.addAlarmAction(
      new SnsAction(
        Topic.fromTopicArn(
          this, 'PagerDutyCPUALARM', PAGERDUTY_SNS
    )))
    diskAlarm.addAlarmAction(
      new SnsAction(
        Topic.fromTopicArn(
          this, 'PagerDutyDISKALARM', PAGERDUTY_SNS
    )))
  }
}
