import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as route53 from '@aws-cdk/aws-route53';
import * as targets from '@aws-cdk/aws-route53-targets';
import * as s3 from '@aws-cdk/aws-s3';
import * as ecr from '@aws-cdk/aws-ecr';
import * as sm from '@aws-cdk/aws-secretsmanager';
import * as rds from '@aws-cdk/aws-rds';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as sqs from '@aws-cdk/aws-sqs';
import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as sns from '@aws-cdk/aws-sns';
import * as subscriptions from '@aws-cdk/aws-sns-subscriptions';
import * as s3n from '@aws-cdk/aws-s3-notifications';
import { Config } from './config';
import { EleriumDataStack } from './elerium-data-stack';
import { ElasticSearch } from './resources/elastic-search';
import { CommonComputeStack } from './common-compute-stack';
import { ElasticCache } from './resources/elastic-cache';
import { ExpandedQueue } from './resources/queue';
import { EleriumComputeStack } from './elerium-compute-stack';
import { CommonStack } from './common-stack';

interface AcsDataStackProps {
  cert: certificateManager.Certificate;
  commonComputeStack: CommonComputeStack;
  commonStack: CommonStack;
  eleriumComputeStack: EleriumComputeStack;
  eleriumDataStack: EleriumDataStack;
  vpc: ec2.Vpc;
}

export class AcsDataStack extends cdk.Stack {
  assetsBucket: s3.Bucket;

  /* ECR Repos */
  ecrCrashLogFileInvalidationService: ecr.Repository;
  ecrDataManager: ecr.Repository;
  ecrAddonService: ecr.Repository;

  // Secrets
  addonServiceApiKey: sm.Secret;
  apiKey: sm.Secret;
  authenticationID: sm.Secret;
  authenticationKey: sm.Secret;
  eleriumApiKey: sm.Secret;
  eleriumDatabaseSecret: rds.DatabaseSecret;
  eleriumDatabaseHostName: sm.Secret;
  encryptionTokenKey: sm.Secret;
  gameManagementApiKey: sm.Secret;
  inApiKey: sm.Secret;
  logServiceApiKey: sm.Secret;

  // Crash Log File Validation Data Infra
  crashLogFileValidationQueue: sqs.Queue;
  crashLogDynamoTable: dynamodb.Table;

  // Data Manager Data Infra
  dataManagerESCluster: ElasticSearch;
  dataManagerRedisCluster: ElasticCache;
  dataManagerQueue: sqs.Queue;

  // Addon Service Data Infra
  addonServiceRedisCluster: ElasticCache;
  crashLoggingS3: s3.Bucket;
  crashLoggingSNS: sns.Topic;

  constructor(scope: cdk.Construct, config: Config, props: AcsDataStackProps) {
    super(scope, config.prefix + 'AcsData', { env: config.env });

    // CloudFront origin access identity
    const originAccessIdentity = new cloudfront.OriginAccessIdentity(this, 'OriginAccessIdentity');

    // Assets bucket
    this.assetsBucket = new s3.Bucket(this, 'Assets', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });
    this.assetsBucket.grantRead(originAccessIdentity);

    // Assets distribution
    const assetsDist = new cloudfront.CloudFrontWebDistribution(this, 'AssetsDist', {
      originConfigs: [
        {
          s3OriginSource: {
            s3BucketSource: this.assetsBucket,
            originAccessIdentity,
          },
          behaviors: [{ isDefaultBehavior: true }],
        },
      ],
      viewerCertificate: cloudfront.ViewerCertificate.fromAcmCertificate(props.cert, {
        aliases: ['client-cdn.overwolf.wtf', 'client.forgecdn.net'],
      }),
      priceClass: cloudfront.PriceClass.PRICE_CLASS_ALL,
    });

    // Assets DNS records
    new route53.ARecord(this, 'AssetsDistARecord', {
      recordName: 'client-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.AaaaRecord(this, 'AssetsDistAaaaRecord', {
      recordName: 'client-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.ARecord(this, 'AssetsDistCdnZoneARecord', {
      recordName: 'client.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.AaaaRecord(this, 'AssetsDistCdnZoneAaaaRecord', {
      recordName: 'client.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    // ECR Repos
    this.ecrCrashLogFileInvalidationService = new ecr.Repository(this, 'crashlog_file_invalidation_service', {
      repositoryName: 'crash_log_file_invalidation_service',
    });
    this.ecrDataManager = new ecr.Repository(this, 'data_manager', {
      repositoryName: 'data_manager',
    });
    this.ecrAddonService = new ecr.Repository(this, 'addon_service', {
      repositoryName: 'addon_service',
    });

    // Secrets
    this.addonServiceApiKey = new sm.Secret(this, 'CrashlogFileInvalidationAddonServiceApiKey', {
      description: 'API Key that allows crash log file invalidation to talk to AddonService',
    });

    this.apiKey = new sm.Secret(this, 'AddonServiceApiKey', {
      description: 'TODO: Figure out what this does',
    });

    this.authenticationID = new sm.Secret(this, 'AddonServiceAuthenticationID', {
      description: 'TODO: Figure out what this does',
    });

    this.authenticationKey = new sm.Secret(this, 'AddonServiceAuthenticationKey', {
      description: 'TODO: Figure out what this does',
    });

    this.eleriumApiKey = new sm.Secret(this, 'DataManagerEleriumApiKey', {
      description: 'API Key that allows Data Manager to talk to Elerium',
    });

    this.eleriumDatabaseSecret = new rds.DatabaseSecret(this, 'AddonServiceEleriumDatabaseSecret', {
      username: 'addons_service',
      masterSecret: props.eleriumDataStack.db.secret,
    });

    this.eleriumDatabaseHostName = new sm.Secret(this, 'AddonServiceEleriumDatabaseHostname', {
      description: 'This is the database hostname for the Elerium DB',
    });

    this.encryptionTokenKey = new sm.Secret(this, 'AddonServiceEncryptionTokenKey', {
      description: 'TODO: Figure out what this does',
    });

    this.gameManagementApiKey = new sm.Secret(this, 'AddonServiceGameManagementKey', {
      description: 'TODO: Figure out what this does',
    });

    this.inApiKey = new sm.Secret(this, 'AddonServiceInApiKey', {
      description: 'TODO: Find out what this does',
    });

    this.logServiceApiKey = new sm.Secret(this, 'AddonServiceLogServiceApiKey', {
      description: 'TODO: Figure out what this does',
    });

    this.dataManagerESCluster = this.setupElasticSearch(config, props);

    this.dataManagerRedisCluster = this.setupDataManagerRedisCluster(config, props.vpc);

    this.addonServiceRedisCluster = this.setupAddonServiceRedisCluster(config, props.vpc);

    // SNS
    this.crashLoggingSNS = new sns.Topic(this, 'CrashLoggingS3Event', {
      displayName: 'S3 Event from CrashLogging Bucket',
    });

    // SQS Queue
    this.crashLogFileValidationQueue = new ExpandedQueue(this, 'CrashLogFileInvalidationQueue');
    this.dataManagerQueue = new ExpandedQueue(this, 'DataManagerQueue');

    // SNS Perms
    this.crashLoggingSNS.addSubscription(new subscriptions.SqsSubscription(this.crashLogFileValidationQueue));
    props.eleriumDataStack.dataManagerInvalidationsSns.addSubscription(
      new subscriptions.SqsSubscription(this.dataManagerQueue)
    );

    // DynamoDB
    this.crashLogDynamoTable = this.setupCrashLogDynamoTable();

    // S3
    this.crashLoggingS3 = new s3.Bucket(this, 'CrashLogging', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });

    this.crashLoggingS3.addEventNotification(
      s3.EventType.OBJECT_CREATED,
      new s3n.SnsDestination(this.crashLoggingSNS),
      {
        prefix: 'ingest/',
      }
    );
  }

  public setupElasticSearch(config: Config, props: AcsDataStackProps) {
    return new ElasticSearch(this, 'DataManagerElasticSearch', config, {
      elasticsearchVersion: '6.3',
      elasticsearchClusterConfig: {
        instanceType: 'r5.4xlarge.elasticsearch',
        instanceCount: 18,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
          availabilityZoneCount: 3,
        },
        dedicatedMasterCount: 3,
        dedicatedMasterEnabled: true,
        dedicatedMasterType: 'c4.2xlarge.elasticsearch',
      },
      ebsOptions: {
        ebsEnabled: true,
        volumeSize: 150,
      },
      vpc: props.vpc,
      vpcOptions: {
        // We have to override this to use publicSubnets
        subnetIds: props.vpc.publicSubnets.map((subnet) => subnet.subnetId).slice(0, 3), // We can only use 3 subnets
      },
    })
      .addInternalIngress()
      .addPeerIngress(props.commonComputeStack.vpnSecurityGroup);
  }

  private setupDataManagerRedisCluster(config: Config, vpc: ec2.Vpc) {
    return new ElasticCache(this, config.prefix + `DataManagerRedisCluster`, {
      cacheNodeType: config.dataManagerElasticServerType,
      replicasPerNodeGroup: config.dataManagerElasticServerCount,
      replicationGroupDescription: 'Data Manager Redis Cluster',
      vpc,
    });
  }

  private setupAddonServiceRedisCluster(config: Config, vpc: ec2.Vpc) {
    return new ElasticCache(this, config.prefix + `Addon Service RedisCluster`, {
      cacheNodeType: config.addonServiceElasticServerType,
      replicasPerNodeGroup: config.addonServiceElasticServerCount,
      replicationGroupDescription: 'Addon Service Redis Cluster',
      vpc,
    });
  }

  private setupCrashLogDynamoTable() {
    const table = new dynamodb.Table(this, 'CrashLogs', {
      tableName: 'Crash_Logs',
      partitionKey: { name: 'projectId', type: dynamodb.AttributeType.NUMBER },
      sortKey: { name: 'id', type: dynamodb.AttributeType.STRING },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
    });

    return table;
  }
}
