import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as route53 from '@aws-cdk/aws-route53';
import { Config } from './config';
import { CommonStack } from './common-stack';

export interface CertProps {
  commonStack: CommonStack;
}

/**
 * Stack for managing SSL certificates since creation/update of them can block stack updates.
 */
export class CertificatesStack extends cdk.Stack {
  cert: certificateManager.Certificate;

  constructor(scope: cdk.Construct, config: Config, props: CertProps) {
    super(scope, config.prefix + 'Certificates', { env: config.env });

    // We can't use the fully automated DnsValidatedCertificate because we use multiple zones and have domains in Cloudflare DNS as well.
    this.cert = new certificateManager.Certificate(this, 'Cert', {
      domainName: 'curseforge.com',
      subjectAlternativeNames: [
        '*.curseforge.com',
        'wowace.com',
        '*.wowace.com',
        'bukkit.org',
        '*.bukkit.org',
        'sc2mapster.com',
        '*.sc2mapster.com',
        'skyrimforge.com',
        '*.skyrimforge.com',
        'forgecdn.net',
        '*.forgecdn.net',
        'forgesvc.net',
        '*.forgesvc.net',
        'overwolf.wtf',
        '*.overwolf.wtf',
      ],
      validationMethod: certificateManager.ValidationMethod.DNS,
    });

    new route53.CnameRecord(this, 'ServiceDnsValidationRecord', {
      zone: props.commonStack.servicesZone,
      recordName: '_0449f65d91cb86560edea537360cc079',
      domainName: '_5b40e0f13807e223d2d9ff16de73fed7.zdxcnfdgtt.acm-validations.aws.',
    });

    new route53.CnameRecord(this, 'CdnDnsValidationRecord', {
      zone: props.commonStack.cdnZone,
      recordName: '_dd9b2939ca0be95d3488442c97c438f9',
      domainName: '_a3a34675f076a0bea74025ba6ab48c7c.zdxcnfdgtt.acm-validations.aws.',
    });

    new route53.CnameRecord(this, 'InternalDnsValidationRecord', {
      zone: props.commonStack.internalZone,
      recordName: '_2c3083d0f4800489c6f1001b8f969973',
      domainName: '_6ec36723305cfce9ed37667728d8d726.jfrzftwwjs.acm-validations.aws.',
    });
  }
}
