import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import { Config } from './config';
import { CfWidgetDataStack } from './cfwidget-data-stack';
import { CommonStack } from './common-stack';
import { Helper } from './resources/helper';
import { EleriumDataStack } from './elerium-data-stack';

export interface CfWidgetComputeStackProps {
  cert: certificateManager.Certificate;
  cluster: ecs.Cluster;
  commonStack: CommonStack;
  dataStack: CfWidgetDataStack;
  eleriumDataStack: EleriumDataStack;
}

export class CFWidgetComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: CfWidgetComputeStackProps) {
    super(scope, config.prefix + 'CfWidgetCompute', { env: config.env });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDef');

    const container = taskDefinition.addContainer('CfWidget', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrRepo),
      memoryLimitMiB: 4096,
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'cfwidget',
        logRetention: 30,
      }),
    });

    container.addPortMappings({ containerPort: 80 });

    const cfWidgetService = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'Service', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: config.cfWidgetDesiredTaskCount,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
      domainName: 'cfwidget.overwolf.wtf',
      domainZone: props.commonStack.internalZone,
      certificate: props.cert,
    });

    cfWidgetService.targetGroup.configureHealthCheck({
      path: '/api/cfwidget/health-check',
      healthyThresholdCount: config.cfWidgetHealthyTaskCount,
      unhealthyThresholdCount: config.cfWidgetUnhealthyTaskCount,
      timeout: cdk.Duration.seconds(2),
      interval: cdk.Duration.seconds(5),
    });

    cfWidgetService.targetGroup.setAttribute('deregistration_delay.timeout_seconds', '10');
    props.eleriumDataStack.databaseSecret.grantRead(taskDefinition.taskRole);
    Helper.addP2PDefaultPortIngress(
      props.eleriumDataStack.db.connections,
      cfWidgetService.cluster.connections,
      undefined,
      false
    );
  }
}
