import * as cdk from '@aws-cdk/core';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as route53 from '@aws-cdk/aws-route53';
import { Config } from './config';
import { ClamAvDataStack } from './clamav-data-stack';

export interface ClamAvComputeStackProps {
  dataStack: ClamAvDataStack;
  cluster: ecs.Cluster;
  zone: route53.HostedZone;
}

export class ClamAvComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: ClamAvComputeStackProps) {
    super(scope, config.prefix + 'ClamAvCompute', { env: config.env });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDef');

    const container = taskDefinition.addContainer('ClamAV', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrRepo),
      memoryLimitMiB: 4096,
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'clamav',
        logRetention: 30,
      }),
    });

    container.addPortMappings({ containerPort: 3310 });

    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      hardLimit: 1048576,
      softLimit: 1048576,
    });

    const clamAvService = new ecsPatterns.NetworkLoadBalancedEc2Service(this, 'Service', {
      cluster: props.cluster,
      memoryLimitMiB: 4096,
      taskDefinition,
      desiredCount: 3,
      domainName: 'clamav.overwolf.wtf',
      domainZone: props.zone,
      listenerPort: 3310,
      publicLoadBalancer: false,
    });

    clamAvService.targetGroup.setAttribute('deregistration_delay.timeout_seconds', '10');
  }
}
