import * as cdk from '@aws-cdk/core';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import { Config } from './config';
import { CommonStack } from './common-stack';

export interface CommonComputeProps {
  vpc: ec2.Vpc;
  commonData: CommonStack;
}

export class CommonComputeStack extends cdk.Stack {
  cluster: ecs.Cluster;
  vpnSecurityGroup: ec2.SecurityGroup;

  constructor(scope: cdk.Construct, config: Config, props: CommonComputeProps) {
    super(scope, config.prefix + 'CommonCompute', { env: config.env });

    this.vpnSecurityGroup = new ec2.SecurityGroup(this, 'VPNSecurityGroup', {
      vpc: props.vpc,
    });

    this.cluster = new ecs.Cluster(this, 'CommonCluster', {
      clusterName: 'Common',
      vpc: props.vpc,
    });

    const ags = this.cluster.addCapacity('CommonClusterScalingCapacity', {
      instanceType: new ec2.InstanceType(config.commonClusterInstanceType),
      desiredCapacity: config.commonClusterCapacity,
    });

    // Because NLB's rewrite the IP to appear to be coming from the service host unless we whitelist each individual hos
    // this is the best way to let an NLB TG access its targets.
    this.cluster.connections.allowFrom(
      ec2.Peer.ipv4(props.vpc.vpcCidrBlock),
      ec2.Port.tcpRange(32768, 65535),
      'Network Load balancers to target'
    );

    // SSM Support
    ags.addUserData(
      'cd /tmp',
      'sudo yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
      'sudo systemctl enable amazon-ssm-agent',
      'sudo systemctl start amazon-ssm-agent'
    );
    ags.grantPrincipal.addToPrincipalPolicy(
      new iam.PolicyStatement({
        actions: ['ssm:UpdateInstanceInformation', 'ec2messages:*', 'ecr:GetAuthorizationToken', 'ssmmessages:*'],
        resources: ['*'],
      })
    );
  }
}
