import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import { IpWhitelistProps } from './resources/ip-whitelist-props';
import { InfrastructureMaintainers } from './ip-whitelists';

export type Config = {
  /** What to prefix stacks with, for example `Prod` would create `ProdVpc` and `ProdCommon` */
  prefix: string;

  /* Pragmatic enviroment name, such as production, staging, or development */
  envName: string;

  /** Settings for the Services/Common cluster that runs most tasks */
  commonClusterCapacity: number;
  commonClusterInstanceType: string;

  /** Settings for the AddonService Cluster */
  acsClusterCapacity: number;

  /** Download Stats settings */
  dlsRedirectHost: string; // Which domain to redirect the client to after counting their download.
  dlsElasticDataType: string;
  dlsElasticDataCount: number;
  dlsElasticMasterType: string; // Count is excluded from config as it should remain 3
  dlsCacheServerCount: number;
  dlsCacheServerType: string;

  /* Elerium */
  eleriumDatabaseSize: ec2.InstanceType;

  /** IP Allowlists */
  ipAllowlist: IpAllowlistConfig;

  /** The CDK environment specifies which region and account to use. */
  env: cdk.Environment;
  prefixList: string;

  /** Data Syncing */
  dataSyncSecurityGroup: string;

  /** Data Manager settings */
  dataManagerElasticServerType: string;
  dataManagerElasticServerCount: number;

  /** Addon Service settings */
  addonServiceElasticServerType: string;
  addonServiceElasticServerCount: number;

  /** URL to handle SES email notifications. */
  sesNotificationsUrl: string;

  /**
   * SES email notifications secret version. Bump this and redeploy EleriumDataStack (e.g. ProdEleriumData) if the secret ever changes.
   *
   * This is a workaround for CloudFormation resources not updating when an injected secret does.
   */
  sesNotificationsSecretVersion: number;

  /** (Optional) Email to send alerts to. */
  alertsEmail?: string;

  /** Bukkit */
  bukkitElasticSearchCount: number;
  bukkitElasticBeanstalkCount: number;
  bukkitAuroraInstanceCount: number;

  /** Legacy ACS */
  legacyACSBeanstalkMinCount: number;
  legacyACSBeanstalkMaxCount: number;

  /** CFWidget */
  cfWidgetDesiredTaskCount: number;
  cfWidgetUnhealthyTaskCount: number;
  cfWidgetHealthyTaskCount: number;

  /** RepoHost */
  repoHostDesiredTaskCount: number;

  /** Files Service */
  filesServiceDesiredTaskCount: number;
  filesServiceUnhealthyTaskCount: number;
  filesServiceHealthyTaskCount: number;

  /** Elerium / CurseForge website */
  eleriumElasticBeanstalkMinCount: number;
  eleriumElasticBeanstalkMaxCount: number;
  eleriumElasticBeanstalkMinInstancesInService: number;
  eleriumElasticBeanstalkHealthyThresholdCount: number;
  eleriumElasticBeanstalkUnhealthyThresholdCount: number;

  /** CurseForge Crash Logs settings */
  curseForgeCrashLogsElasticServerType: string;
  curseForgeCrashLogsElasticServerCount: number;
};

/* Split out of the main config to maintain readability */
export type IpAllowlistConfig = {
  InternalDataStores: IpWhitelistProps[];
};

// Pass in lists of constant IP addresses from ip-whitelists. For example, you would do InfrastructureMaintainers.concat(ExampleSecondaryGroup) to add both groups to InternalDataStores
const ALLOWLISTS: IpAllowlistConfig[] = [
  {
    InternalDataStores: InfrastructureMaintainers,
  },
];

export const CONFIGS: Config[] = [
  {
    prefix: 'Prod',
    envName: 'Production',
    commonClusterCapacity: 6,
    commonClusterInstanceType: 'c5.2xlarge',
    acsClusterCapacity: 20,
    dlsRedirectHost: 'https://media.forgecdn.net',
    dlsCacheServerCount: 2,
    dlsCacheServerType: 'cache.m4.large',
    dlsElasticDataCount: 6,
    dlsElasticDataType: 'r4.xlarge.elasticsearch',
    dlsElasticMasterType: 'c4.xlarge.elasticsearch',
    eleriumDatabaseSize: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE4), // ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.XLARGE16)
    ipAllowlist: ALLOWLISTS[0],
    env: {
      account: '810559446296',
      region: 'us-east-1',
    },
    prefixList: 'pl-4e2ece27',
    dataSyncSecurityGroup: 'sg-00d00e6f030aed7b5',
    dataManagerElasticServerType: 'cache.m4.large',
    dataManagerElasticServerCount: 3,
    addonServiceElasticServerType: 'cache.m4.large',
    addonServiceElasticServerCount: 3,
    curseForgeCrashLogsElasticServerType: 'cache.m4.large',
    curseForgeCrashLogsElasticServerCount: 3,
    sesNotificationsUrl: 'https://elerium.overwolf.wtf/api/email-notifications',
    sesNotificationsSecretVersion: 1,
    bukkitElasticSearchCount: 3,
    bukkitElasticBeanstalkCount: 3,
    bukkitAuroraInstanceCount: 2,
    legacyACSBeanstalkMinCount: 2,
    legacyACSBeanstalkMaxCount: 3,
    cfWidgetDesiredTaskCount: 3,
    cfWidgetUnhealthyTaskCount: 2,
    cfWidgetHealthyTaskCount: 3,
    repoHostDesiredTaskCount: 3,
    filesServiceDesiredTaskCount: 3,
    filesServiceUnhealthyTaskCount: 2,
    filesServiceHealthyTaskCount: 3,
    eleriumElasticBeanstalkMinCount: 3,
    eleriumElasticBeanstalkMaxCount: 9,
    eleriumElasticBeanstalkMinInstancesInService: 3,
    eleriumElasticBeanstalkHealthyThresholdCount: 3,
    eleriumElasticBeanstalkUnhealthyThresholdCount: 2,
  },
];

export const BUKKIT_LEGACY_ACC_ID = '526403043015';
export const BUKKIT_LEGACY_DATA_BUCKET = 'arn:aws:s3:::bukkit-forums-backups';
export const ELERIUM_LEGACY_DATA_BUCKET =
  'arn:aws:s3:::sql-scratch-space-default-useast1-79b0cfe9fbdf1386e0b9d32483224';
export const ELERIUM_LEGACY_ASSETS_BUCKET =
  'arn:aws:s3:::elerium-media-production-useast1-05897d3b7eb47d49d45163ff37555d';
export const ACS_LEGACY_ASSETS_BUCKET =
  'arn:aws:s3:::addons-service-production-useast1-ddfc94577feea2418c1a238f295a3';
export const ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET = 'arn:aws:s3:::crashlogging-prods3017d3f73-sgqbup58f1zx';
export const ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET =
  'arn:aws:s3:::mod-loaders-production-useast1-d74aef4d7091eb4a2d0a6b430f7913a6';
