import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as sqs from '@aws-cdk/aws-sqs';
import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as iam from '@aws-cdk/aws-iam';

import { Config } from './config';
import { ElasticSearch } from './resources/elastic-search';
import { ExpandedQueue } from './resources/queue';
import { ElasticCache } from './resources/elastic-cache';
import { CommonComputeStack } from './common-compute-stack';

export interface DlsDataStackProps {
  vpc: ec2.Vpc;
  commonComputeStack: CommonComputeStack;
}

export class DlsDataStack extends cdk.Stack {
  elasticsearch: ElasticSearch;
  redis: ElasticCache;

  /* ECR Repos */
  ecrDownloadStatsEdge: ecr.Repository;
  ecrDownloadStats: ecr.Repository;
  ecrDownloadStatsJobs: ecr.Repository;
  ecrDownloadIngest: ecr.Repository;

  /* SQS Queues */
  redisQueue: sqs.Queue;
  elasticQueue: sqs.Queue;
  dynamoQueue: sqs.Queue;

  /* Dynamo */
  dynamoLegacyTable: dynamodb.Table;
  dynamoTable: dynamodb.Table;
  dynamoProjectTable: dynamodb.Table;
  dynamoProjectFilesTable: dynamodb.Table;

  constructor(scope: cdk.Construct, config: Config, props: DlsDataStackProps) {
    super(scope, config.prefix + 'DlsData', { env: config.env });

    this.elasticsearch = this.setupElasticSearch(config, props);
    this.redis = this.setupElasticCache(config, props);

    this.ecrDownloadStatsEdge = new ecr.Repository(this, 'download_stats_edge', {
      repositoryName: 'download_stats_edge',
    });
    this.ecrDownloadStatsJobs = new ecr.Repository(this, 'download_stats_jobs', {
      repositoryName: 'download_stats_jobs',
    });
    this.ecrDownloadStats = new ecr.Repository(this, 'download_stats', { repositoryName: 'download_stats' });
    this.ecrDownloadIngest = new ecr.Repository(this, 'download_ingest', { repositoryName: 'download_ingest' });

    this.redisQueue = new ExpandedQueue(this, 'RedisIngest');
    this.elasticQueue = new ExpandedQueue(this, 'ElasticIngest');
    this.dynamoQueue = new ExpandedQueue(this, 'DynamoIngest');

    this.dynamoLegacyTable = this.setupLegacyRecordsDynamo();
    this.dynamoTable = this.setupRecordsDynamo();
    this.dynamoProjectTable = this.setupRecordsProjectDynamo();
    this.dynamoProjectFilesTable = this.setupRecordsProjectFilesDynamo();
    this.setupRecordsDynamoTest();
  }

  private setupLegacyRecordsDynamo() {
    const table = new dynamodb.Table(this, 'DlsLegacyRecords', {
      tableName: 'DLS_Records_Legacy',
      partitionKey: { name: 'Guid', type: dynamodb.AttributeType.STRING },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
    });

    table.addGlobalSecondaryIndex({
      indexName: 'FileID_Timestamp',
      partitionKey: { name: 'FileID', type: dynamodb.AttributeType.NUMBER },
      sortKey: { name: 'Timestamp', type: dynamodb.AttributeType.NUMBER },
    });

    table.addGlobalSecondaryIndex({
      indexName: 'Timestamp',
      partitionKey: { name: 'Timestamp', type: dynamodb.AttributeType.NUMBER },
    });

    return table;
  }

  private setupRecordsProjectDynamo() {
    const dynamoTable = new dynamodb.Table(this, 'DlsProjects', {
      pointInTimeRecovery: true,
      partitionKey: { name: 'ProjectID', type: dynamodb.AttributeType.NUMBER },
      sortKey: { name: 'GameID', type: dynamodb.AttributeType.NUMBER },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST, // Attempting on-demand on this table, may need to be swapped to provisioned. To swap change the above and uncomment the below read/write autoscaling.
    });

    return dynamoTable;
  }

  private setupRecordsProjectFilesDynamo() {
    const dynamoTable = new dynamodb.Table(this, 'DlsProjectsFiles', {
      pointInTimeRecovery: true,
      partitionKey: { name: 'ID', type: dynamodb.AttributeType.NUMBER },
      sortKey: { name: 'ProjectID', type: dynamodb.AttributeType.NUMBER },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST, // Attempting on-demand on this table, may need to be swapped to provisioned. To swap change the above and uncomment the below read/write autoscaling.
    });

    return dynamoTable;
  }

  private setupRecordsDynamoTest() {
    const dynamoTable = new dynamodb.Table(this, 'DlsRecordsTest', {
      tableName: 'DLS_Records_Test_With_Sort',
      pointInTimeRecovery: true,
      partitionKey: { name: 'Guid', type: dynamodb.AttributeType.STRING },
      sortKey: { name: 'FileID', type: dynamodb.AttributeType.NUMBER },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST, // Attempting on-demand on this table, may need to be swapped to provisioned. To swap change the above and uncomment the below read/write autoscaling.
    });

    return dynamoTable;
  }

  private setupRecordsDynamo() {
    const dynamoTable = new dynamodb.Table(this, 'DlsRecords', {
      tableName: 'DLS_Records_New',
      pointInTimeRecovery: true,
      partitionKey: { name: 'Guid', type: dynamodb.AttributeType.STRING },
      billingMode: dynamodb.BillingMode.PAY_PER_REQUEST, // Attempting on-demand on this table, may need to be swapped to provisioned. To swap change the above and uncomment the below read/write autoscaling.
    });

    // const readScaling = dynamoTable.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 2000 });

    // readScaling.scaleOnUtilization({
    //   targetUtilizationPercent: 85,
    // });

    // const writeScaling = dynamoTable.autoScaleWriteCapacity({ minCapacity: 100, maxCapacity: 3000 });

    // writeScaling.scaleOnUtilization({
    //   targetUtilizationPercent: 85,
    // });

    dynamoTable.addGlobalSecondaryIndex({
      indexName: 'FileID_Timestamp',
      partitionKey: { name: 'FileID', type: dynamodb.AttributeType.NUMBER },
      sortKey: { name: 'Timestamp', type: dynamodb.AttributeType.NUMBER },
    });

    dynamoTable.addGlobalSecondaryIndex({
      indexName: 'Timestamp',
      partitionKey: { name: 'Timestamp', type: dynamodb.AttributeType.NUMBER },
    });

    return dynamoTable;
  }

  private setupElasticCache(config: Config, props: DlsDataStackProps) {
    return new ElasticCache(this, config.prefix + `DlsDataCache`, {
      engineVersion: '3.2.4',
      cacheNodeType: config.dlsCacheServerType,
      replicasPerNodeGroup: config.dlsCacheServerCount,
      replicationGroupDescription: 'DLS Cache for up to date downloads per project file',
      vpc: props.vpc,
    });
  }

  private setupElasticSearch(config: Config, props: DlsDataStackProps) {
    return new ElasticSearch(this, config.prefix + 'DlsDataSearch', config, {
      accessPolicies: new iam.PolicyDocument({
        // This is a VPC Cluster, don't do this for public clusters
        statements: [
          new iam.PolicyStatement({
            resources: ['*'],
            principals: [new iam.AnyPrincipal()],
            actions: ['es:*'],
          }),
        ],
      }),
      domainName: config.prefix + '-dls-data',
      ebsOptions: {
        ebsEnabled: true,
        volumeSize: 200,
        volumeType: 'gp2',
      },
      elasticsearchClusterConfig: {
        dedicatedMasterCount: 3,
        dedicatedMasterEnabled: true,
        dedicatedMasterType: config.dlsElasticMasterType,
        instanceCount: config.dlsElasticDataCount,
        instanceType: config.dlsElasticDataType,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
          availabilityZoneCount: 3,
        },
      },
      elasticsearchVersion: '6.2',
      vpc: props.vpc,
      vpcOptions: {
        // We have to override this to use publicSubnets
        subnetIds: props.vpc.publicSubnets.map((subnet) => subnet.subnetId).slice(0, 3), // We can only use 3 subnets
      },
    })
      .addInternalIngress()
      .addPeerIngress(props.commonComputeStack.vpnSecurityGroup);
  }
}
