import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as iam from '@aws-cdk/aws-iam';
import * as elasticbeanstalk from '@aws-cdk/aws-elasticbeanstalk';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53';
import { Config } from './config';
import { EleriumDataStack } from './elerium-data-stack';
import { CommonStack } from './common-stack';
import { Helper } from './resources/helper';
import { ElasticBeanstalkTarget } from './resources/elastic-beanstalk-target';

export interface EleriumComputeProps {
  dataStack: EleriumDataStack;
  commonStack: CommonStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class EleriumComputeStack extends cdk.Stack {
  // Stack Assets
  eleriumEbSecurityGroup: ec2.SecurityGroup;
  eleriumIamInstanceRole: iam.Role;
  eleriumControlDomain: route53.CnameRecord;

  constructor(scope: cdk.Construct, config: Config, props: EleriumComputeProps) {
    super(scope, config.prefix + 'EleriumCompute', { env: config.env });

    // Production Forum
    const eleriumIamRole = new iam.Role(this, 'EleriumEnvServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    this.eleriumIamInstanceRole = new iam.Role(this, 'EleriumEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
      ],
      inlinePolicies: {
        // These are required for the beanstalk extension scripts to function
        beanstalk_extensions: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: [
                'ec2:DisassociateAddress',
                'ec2:DescribeAddresses',
                'ec2:AssociateAddress',
                'ec2:DescribeTags',
              ],
              resources: ['*'],
            }),
          ],
        }),
        secret_manager: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['secretsmanager:Describe*', 'secretsmanager:Get*', 'secretsmanager:List*'],
              resources: [
                props.dataStack.databaseSecret.secretArn,
                props.dataStack.globalSecretProvider.secretArn,
                props.dataStack.smtpSecret.secretArn,
                props.dataStack.recaptchaSecret.secretArn,
                props.dataStack.textProviderSecret.secretArn,
                props.dataStack.twitchClientSecrets.secretArn,
              ],
            }),
          ],
        }),
        // There is no way to limit this and still send SMS....
        sms: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['sns:Publish'],
              resources: ['*'],
            }),
          ],
        }),
      },
    });

    const eleriumIamInstanceProfile = new iam.CfnInstanceProfile(this, 'EleriumEnvInstanceRoleProfile', {
      roles: [this.eleriumIamInstanceRole.roleName],
    });

    eleriumIamInstanceProfile.node.addDependency(this.eleriumIamInstanceRole);

    this.eleriumEbSecurityGroup = new ec2.SecurityGroup(this, 'EleriumBeanstalkSecurityGroup', {
      vpc: props.vpc,
    });

    const eleriumLbSecurityGroup = new ec2.SecurityGroup(this, 'EleriumLoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    // Used for the invalidation manager
    const eleriumCrosstalkSecurityGroup = new ec2.SecurityGroup(this, 'EleriumCrosstalkSecurityGroup', {
      vpc: props.vpc,
    });

    eleriumCrosstalkSecurityGroup.addIngressRule(this.eleriumEbSecurityGroup, ec2.Port.tcp(80));

    const app = new elasticbeanstalk.CfnApplication(this, 'BeanstalkApp', {
      applicationName: 'elerium',
      resourceLifecycleConfig: {
        serviceRole: eleriumIamRole.roleArn,
      },
    });

    app.node.addDependency(eleriumIamRole);

    const env = new elasticbeanstalk.CfnEnvironment(this, 'BeanstalkEnv', {
      applicationName: app.applicationName!,
      solutionStackName: '64bit Windows Server Core 2016 v2.5.8 running IIS 10.0',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.4xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: config.eleriumElasticBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: config.eleriumElasticBeanstalkMaxCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Bucket',
          value: props.commonStack.accessLogs.bucketName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Enabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'application',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: eleriumIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: eleriumIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'DeploymentPolicy',
          value: 'Rolling',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSizeType',
          value: 'Fixed',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MaxBatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MinInstancesInService',
          value: config.eleriumElasticBeanstalkMinInstancesInService.toString(),
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateType',
          value: 'Health',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckInterval',
          value: '5',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthyThresholdCount',
          value: config.eleriumElasticBeanstalkHealthyThresholdCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'UnhealthyThresholdCount',
          value: config.eleriumElasticBeanstalkUnhealthyThresholdCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'MonitoringInterval',
          value: '1 Minute',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeSize',
          value: '50',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeType',
          value: 'gp2',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: [
            this.eleriumEbSecurityGroup.securityGroupName,
            eleriumCrosstalkSecurityGroup.securityGroupName,
          ].join(','),
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'SecurityGroups',
          value: eleriumLbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'Protocol',
          value: 'HTTPS',
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'SSLCertificateArns',
          value: props.cert.certificateArn,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'AssociatePublicIpAddress',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckPath',
          value: '/health-check',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'MatcherHTTPCode',
          value: '200',
        },
        {
          namespace: 'aws:elasticbeanstalk:hostmanager',
          optionName: 'LogPublicationControl',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'IgnoreHealthCheck',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'StickinessEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckTimeout',
          value: '2',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'IdleTimeout',
          value: '600',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'ManagedActionsEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'PreferredStartTime',
          value: 'Thu:10:10',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'UpdateLevel',
          value: 'minor',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'InstanceRefreshEnabled',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:healthreporting:system',
          optionName: 'SystemType',
          value: 'enhanced',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'MEDIA_BASE_URL',
          value: 'https://' + props.dataStack.ELERIUM_ASSETS_CDN_URL,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'S3_Bucket_Name',
          value: props.dataStack.assetsBucket.bucketName,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_USERNAME_DEFAULT',
          value: props.dataStack.db.secret!.secretValueFromJson('username').toString(), // TODO temporary use admin creds
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_PASSWORD_DEFAULT',
          value: props.dataStack.db.secret!.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_HOSTNAME_DEFAULT',
          value: props.dataStack.db.secret!.secretValueFromJson('host').toString(),
        },
      ],
    });

    env.addDependsOn(app);
    env.node.addDependency(eleriumIamRole);
    env.node.addDependency(eleriumIamInstanceProfile);

    Helper.addDefaultPortIngress(props.dataStack.db.connections, this.eleriumEbSecurityGroup);
    props.dataStack.assetsBucket.grantPut(this.eleriumIamInstanceRole);
    props.dataStack.assetsBucket.grantReadWrite(this.eleriumIamInstanceRole);
    props.dataStack.dataManagerInvalidationsSns.grantPublish(this.eleriumIamInstanceRole);
    props.dataStack.redis.addIngressPeer(this.eleriumEbSecurityGroup.connections, true);

    new route53.ARecord(this, 'EleriumRecord', {
      target: route53.RecordTarget.fromAlias(new ElasticBeanstalkTarget(env)),
      zone: props.commonStack.internalZone,
    });

    this.eleriumControlDomain = new route53.CnameRecord(this, 'EleriumSubdomainRecord', {
      domainName: env.attrEndpointUrl,
      zone: props.commonStack.internalZone,
      recordName: 'elerium',
    });
  }
}
