import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import { Config } from './config';
import { FilesServiceDataStack } from './files-service-data-stack';
import { CommonStack } from './common-stack';
import { EleriumDataStack } from './elerium-data-stack';
import { Helper } from './resources/helper';

export interface FilesServiceComputeStackProps {
  cert: certificateManager.Certificate;
  dataStack: FilesServiceDataStack;
  eleriumDataStack: EleriumDataStack;
  cluster: ecs.Cluster;
  commonStack: CommonStack;
}

export class FilesServiceComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: FilesServiceComputeStackProps) {
    super(scope, config.prefix + 'FilesServiceCompute', { env: config.env });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDef');

    const container = taskDefinition.addContainer('FilesService', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.repo),
      memoryLimitMiB: 1024,
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'FilesService',
        logRetention: 30,
      }),
    });

    container.addPortMappings({ containerPort: 80 });

    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      hardLimit: 1048576,
      softLimit: 1048576,
    });

    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'Service', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: config.filesServiceDesiredTaskCount,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
      domainName: 'files-service.overwolf.wtf',
      domainZone: props.commonStack.internalZone,
      certificate: props.cert,
    });

    service.targetGroup.configureHealthCheck({
      path: '/api/health',
      healthyThresholdCount: config.filesServiceHealthyTaskCount,
      unhealthyThresholdCount: config.filesServiceUnhealthyTaskCount,
      timeout: cdk.Duration.seconds(2),
      interval: cdk.Duration.seconds(5),
    });

    service.targetGroup.setAttribute('deregistration_delay.timeout_seconds', '10');

    props.eleriumDataStack.databaseSecret.grantRead(taskDefinition.taskRole);
    Helper.addP2PDefaultPortIngress(
      props.eleriumDataStack.db.connections,
      service.cluster.connections,
      undefined,
      false
    );
  }
}
