import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as logs from '@aws-cdk/aws-logs';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { Helper } from './resources/helper';
import { CommonComputeStack } from './common-compute-stack';
import { HyruleDataStack } from './hyrule-data-stack';

export interface HyruleComputeProps {
  dataStack: HyruleDataStack;
  commonStack: CommonStack;
  commonCompute: CommonComputeStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class HyruleComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: HyruleComputeProps) {
    super(scope, config.prefix + 'HyruleCompute', { env: config.env });

    this.setupHyrule(config, props);
  }

  private setupHyrule(config: Config, props: HyruleComputeProps) {
    const logGroup = new logs.LogGroup(this, 'HyruleLogGroup', {
      logGroupName: 'Hyrule',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'HyruleTask');

    const envVariables = {
      MYSQL_HOST: props.dataStack.databaseCluster.clusterEndpoint.hostname,
      MYSQL_USERNAME: props.dataStack.databaseCluster.secret!.secretValueFromJson('username').toString(),
      MYSQL_PASSWORD: props.dataStack.databaseCluster.secret!.secretValueFromJson('password').toString(),
      MYSQL_DATABASE: 'hyrule',
    };

    taskDefinition
      .addContainer('Hyrule', {
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrHyrule),
        logging: new ecs.AwsLogDriver({
          streamPrefix: 'Task',
          logGroup,
        }),
        memoryReservationMiB: 1024,
        environment: envVariables,
      })
      .addPortMappings({ containerPort: 8080 });

    taskDefinition.addToTaskRolePolicy(
      new iam.PolicyStatement({
        actions: ['rekognition:*', 'comprehend:*'],
        effect: iam.Effect.ALLOW,
        resources: ['*'],
      })
    );

    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'HyruleService', {
      cluster: props.commonCompute.cluster,
      taskDefinition,
      desiredCount: 1,
      minHealthyPercent: 0,
      maxHealthyPercent: 100,
      certificate: props.cert,
      openListener: false,
      domainZone: props.commonStack.internalZone,
      domainName: 'hyrule.overwolf.wtf',
    });

    service.targetGroup.configureHealthCheck({
      path: '/_health/',
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.s3Bucket.grantReadWrite(taskDefinition.taskRole);
    props.dataStack.eleriumApiSecret.grantRead(taskDefinition.taskRole);
    props.dataStack.googleSecret.grantRead(taskDefinition.taskRole);
    Helper.addP2PDefaultPortIngress(props.dataStack.databaseCluster.connections, service.service.connections);

    // Ingest
    service.loadBalancer.connections.allowFrom(ec2.Peer.prefixList(config.prefixList), ec2.Port.tcp(443));
  }
}
