import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as iam from '@aws-cdk/aws-iam';
import * as elasticbeanstalk from '@aws-cdk/aws-elasticbeanstalk';
import * as ec2 from '@aws-cdk/aws-ec2';
import { Config } from './config';
import { EleriumDataStack } from './elerium-data-stack';
import { CommonStack } from './common-stack';
import { Helper } from './resources/helper';
import * as sm from '@aws-cdk/aws-secretsmanager';

export interface PackagerComputeProps {
  eleriumDataStack: EleriumDataStack;
  commonStack: CommonStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class PackagerComputeStack extends cdk.Stack {
  packagerEbSecurityGroup: ec2.SecurityGroup;
  packagerIamInstanceRole: iam.Role;
  globalProviderSecret: sm.Secret;
  databaseSecret: sm.Secret;

  constructor(scope: cdk.Construct, config: Config, props: PackagerComputeProps) {
    super(scope, config.prefix + 'PackagerCompute', { env: config.env });

    this.databaseSecret = new sm.Secret(this, 'PackagerApplicationDBSecret', {
      description: 'Production Creds for Elerium RDS Instance for Elerium Packager Application',
      secretName: 'EleriumPackager/' + config.envName + '/RDS',
    });

    // These are json secret that needs to be filled in manually.
    this.globalProviderSecret = new sm.Secret(this, 'PackagerGlobalProviderSecret', {
      description: 'Cobalt global provider api key and crypto key',
      secretName: 'EleriumPackager/' + config.envName + '/Global-Provider-Secret',
    });

    // Production Forum
    const packagerIamRole = new iam.Role(this, 'PackagerEnvServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    this.packagerIamInstanceRole = new iam.Role(this, 'PackagerEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
      ],
      inlinePolicies: {
        secret_manager: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['secretsmanager:Describe*', 'secretsmanager:Get*', 'secretsmanager:List*'],
              resources: [
                this.databaseSecret.secretArn,
                this.globalProviderSecret.secretArn,
                props.eleriumDataStack.smtpSecret.secretArn,
                props.eleriumDataStack.recaptchaSecret.secretArn,
                props.eleriumDataStack.textProviderSecret.secretArn,
                props.eleriumDataStack.twitchClientSecrets.secretArn,
              ],
            }),
          ],
        }),
        cloudwatch: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogGroups',
                'logs:DescribeLogStreams',
                'logs:PutRetentionPolicy',
              ],
              resources: ['*'],
            }),
          ],
        }),
      },
    });

    const packagerIamInstanceProfile = new iam.CfnInstanceProfile(this, 'PackagerEnvInstanceRoleProfile', {
      roles: [this.packagerIamInstanceRole.roleName],
    });

    packagerIamInstanceProfile.node.addDependency(this.packagerIamInstanceRole);

    this.packagerEbSecurityGroup = new ec2.SecurityGroup(this, 'PackagerBeanstalkSecurityGroup', {
      vpc: props.vpc,
    });

    const packagerLbSecurityGroup = new ec2.SecurityGroup(this, 'PackagerLoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    const app = new elasticbeanstalk.CfnApplication(this, 'BeanstalkApp', {
      applicationName: 'packager',
      resourceLifecycleConfig: {
        serviceRole: packagerIamRole.roleArn,
      },
    });

    app.node.addDependency(packagerIamRole);

    const env = new elasticbeanstalk.CfnEnvironment(this, 'BeanstalkEnv', {
      applicationName: app.applicationName!,
      solutionStackName: '64bit Windows Server Core 2016 v2.5.8 running IIS 10.0',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: '2',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: '2',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Bucket',
          value: props.commonStack.accessLogs.bucketName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Enabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'application',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBScheme',
          value: 'internal',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: packagerIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: packagerIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'DeploymentPolicy',
          value: 'AllAtOnce',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateEnabled',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckInterval',
          value: '5',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthyThresholdCount',
          value: '3',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'UnhealthyThresholdCount',
          value: '2',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'MonitoringInterval',
          value: '1 Minute',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeSize',
          value: '50',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeType',
          value: 'gp2',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: this.packagerEbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'SecurityGroups',
          value: packagerLbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'Protocol',
          value: 'HTTPS',
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'SSLCertificateArns',
          value: props.cert.certificateArn,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'AssociatePublicIpAddress',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckPath',
          value: '/health-check',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'MatcherHTTPCode',
          value: '200',
        },
        {
          namespace: 'aws:elasticbeanstalk:hostmanager',
          optionName: 'LogPublicationControl',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'IgnoreHealthCheck',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'StickinessEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckTimeout',
          value: '2',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'IdleTimeout',
          value: '600',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'ManagedActionsEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'PreferredStartTime',
          value: 'Thu:10:10',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'UpdateLevel',
          value: 'minor',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'InstanceRefreshEnabled',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:healthreporting:system',
          optionName: 'SystemType',
          value: 'enhanced',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'Packager_WorkingDirectory',
          value: 'C:\\Packager',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'MEDIA_BASE_URL',
          value: 'https://' + props.eleriumDataStack.ELERIUM_ASSETS_CDN_URL,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'S3_Bucket_Name',
          value: props.eleriumDataStack.assetsBucket.bucketName,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_USERNAME_DEFAULT',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('username').toString(), // TODO temporary use admin creds
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_PASSWORD_DEFAULT',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_HOSTNAME_DEFAULT',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('host').toString(),
        },
      ],
    });

    env.addDependsOn(app);
    env.node.addDependency(packagerIamRole);
    env.node.addDependency(packagerIamInstanceProfile);

    Helper.addDefaultPortIngress(props.eleriumDataStack.db.connections, this.packagerEbSecurityGroup);
    props.eleriumDataStack.assetsBucket.grantPut(this.packagerIamInstanceRole);
    props.eleriumDataStack.assetsBucket.grantReadWrite(this.packagerIamInstanceRole);
  }
}
