import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as logs from '@aws-cdk/aws-logs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import { Config } from './config';
import { RepoHostDataStack } from './repo-host-data-stack';
import { CommonStack } from './common-stack';

export interface RepoHostComputeProps {
  vpc: ec2.Vpc;
  cluster: ecs.Cluster;
  dataStack: RepoHostDataStack;
  commonStack: CommonStack;
  cert: certificateManager.Certificate;
}

export class RepoHostComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: RepoHostComputeProps) {
    super(scope, config.prefix + 'RepoHostCompute', { env: config.env });

    this.setupRepoHostProduction(config, props);
  }

  private setupRepoHostProduction(config: Config, props: RepoHostComputeProps) {
    const logGroup = new logs.LogGroup(this, 'RepoHostLogGroup', {
      logGroupName: 'RepoHost',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'RepoHostTask', {
      volumes: [
        {
          name: 'repos',
          efsVolumeConfiguration: {
            fileSystemId: props.dataStack.efsRepoHost.fileSystemId,
          },
        },
      ],
    });

    const container = taskDefinition.addContainer('RepoHostContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrRepoHost),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 1024,
      environment: {
        'RepoHostJob:Env': config.envName,
      },
    });

    container.addPortMappings({ containerPort: 20000 });
    container.addMountPoints({
      sourceVolume: 'repos',
      containerPath: '/media/data',
      readOnly: false,
    });

    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'RepoHostService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: config.repoHostDesiredTaskCount,
      certificate: props.cert,
      domainZone: props.commonStack.internalZone,
      domainName: 'repos',
    });

    service.targetGroup.configureHealthCheck({
      path: '/health-check',
    });
  }
}
