import * as cw from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as es from '@aws-cdk/aws-elasticsearch';
import * as cdk from '@aws-cdk/core';
import { ElasticSearchResourceProps } from './elastic-search-resource-props';
import { Config } from '../config';

export class ElasticSearch extends es.CfnDomain {
  public totalNodeCount: number;
  private securityGroup?: ec2.SecurityGroup;
  private vpc: ec2.IVpc;
  private config: Config;

  constructor(scope: cdk.Stack, name: string, config: Config, props: ElasticSearchResourceProps) {
    super(scope, name, {
      advancedOptions: {
        'rest.action.multi.allow_explicit_index': 'true', // Allow a search request to specify the index, otherwise it falls back to perms
      },
      snapshotOptions: {
        automatedSnapshotStartHour: 7, // 24HR UTC time
      },
      vpcOptions: {
        subnetIds: props.vpc.privateSubnets.map((subnet) => subnet.subnetId).slice(0, 3), // We can only use 3 subnets
      },
      ...(props as es.CfnDomainProps),
      domainName: props.domainName ? props.domainName.toLowerCase() : undefined, // Force domain name to lower case
    });

    this.accessPolicies = this.accessPolicies || {
      Statement: [
        {
          Action: 'es:*',
          Effect: 'Allow',
          Principal: {
            AWS: '*',
          },
          Resource: `arn:aws:es:${config.env.region}:${config.env.account}:domain/*/*`,
        },
      ],
    };

    this.vpc = props.vpc;

    const clusterProps = this.elasticsearchClusterConfig as es.CfnDomain.ElasticsearchClusterConfigProperty;

    // vpc is not part of their properties object so we need to remove it so the object compiles properly
    // unsure if still needed.
    // delete this.properties.vpc;

    this.totalNodeCount = (clusterProps.dedicatedMasterCount as number) + (clusterProps.instanceCount as number);
    this.config = config;
  }

  public addInternalIngress() {
    this.createSecurityGroup();

    this.securityGroup!.addIngressRule(ec2.Peer.ipv4(this.vpc.vpcCidrBlock), ec2.Port.tcp(443));
    return this;
  }

  public addPeerIngress(peer: ec2.IPeer) {
    this.createSecurityGroup();
    this.securityGroup!.addIngressRule(peer, ec2.Port.tcp(443));
    return this;
  }

  // public addAlias(zone: route53.IHostedZone, name: string, recordName: string, ttl?: number) {
  //   Route53Helpers.createCnameRecord(this, zone, name, recordName, this.domainEndpoint, ttl);

  //   return this;
  // }

  public metricWriteIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('WriteIOPS', { statistic: 'sum', ...props });
  }

  public metricReadIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ReadIOPS', { statistic: 'sum', ...props });
  }

  public metric2xxIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('2xx', { statistic: 'sum', color: '#2ca02c', ...props });
  }

  public metric3xxIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('3xx', { statistic: 'sum', color: '#ff7f0e', ...props });
  }

  public metric4xxIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('4xx', { statistic: 'sum', ...props });
  }

  public metric5xxIops(props?: cw.CommonMetricOptions) {
    return this.baseMetric('5xx', { statistic: 'sum', color: '#d62728', ...props });
  }

  public metricClusterStatusRed(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ClusterStatus.red', { statistic: 'max', color: '#d62728', ...props });
  }

  public metricClusterStatusYellow(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ClusterStatus.yellow', { statistic: 'max', color: '#dbdb8d', ...props });
  }

  public metricClusterStatusGreen(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ClusterStatus.green', { statistic: 'min', color: '#2ca02c', ...props });
  }

  public metricElasticsearchRequests(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ElasticsearchRequests', { statistic: 'sum', ...props });
  }

  public metricSearchRate(props?: cw.CommonMetricOptions) {
    return this.baseMetric('SearchRate', { statistic: 'sum', ...props });
  }

  public metricReadLatency(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ReadLatency', { statistic: 'max', ...props });
  }

  public metricWriteLatency(props?: cw.CommonMetricOptions) {
    return this.baseMetric('WriteLatency', { statistic: 'max', ...props });
  }

  public metricSearchLatency(props?: cw.CommonMetricOptions) {
    return this.baseMetric('SearchLatency', { statistic: 'max', ...props });
  }

  public metricDeletedDocuments(props?: cw.CommonMetricOptions) {
    return this.baseMetric('DeletedDocuments', { statistic: 'sum', ...props });
  }

  public metricMasterCPUUtilization(props?: cw.CommonMetricOptions) {
    return this.baseMetric('MasterCPUUtilization', { statistic: 'max', ...props });
  }

  public metricMasterReachableFromNode(props?: cw.CommonMetricOptions) {
    return this.baseMetric('MasterReachableFromNode', { statistic: 'min', ...props });
  }

  public metricMasterJVMMemoryPressure(props?: cw.CommonMetricOptions) {
    return this.baseMetric('MasterJVMMemoryPressure', { statistic: 'max', ...props });
  }

  public metricDataCPUUtilization(props?: cw.CommonMetricOptions) {
    return `SEARCH('{AWS/ES,ClientId,DomainName,NodeId} ${this.domainName} MetricName=\"CPUUtilization\"', 'Maximum', 300)`;
  }

  public metricDataFreeStorageSpace(props?: cw.CommonMetricOptions) {
    return `SEARCH('{AWS/ES,ClientId,DomainName,NodeId} ${this.domainName} MetricName=\"FreeStorageSpace\"', 'Maximum', 300)`;
  }

  public metricDataJVMMemoryPressure(props?: cw.CommonMetricOptions) {
    return `SEARCH('{AWS/ES,ClientId,DomainName,NodeId} ${this.domainName} MetricName=\"JVMMemoryPressure\"', 'Maximum', 300)`;
  }

  public metricClusterIndexWritesBlocked(props?: cw.CommonMetricOptions) {
    return this.baseMetric('ClusterIndexWritesBlocked', { statistic: 'sum', ...props });
  }

  public metricNodes(props?: cw.CommonMetricOptions) {
    return this.baseMetric('Nodes', { statistic: 'min', ...props });
  }

  public metricAutomatedSnapshotFailure(props?: cw.CommonMetricOptions) {
    return this.baseMetric('AutomatedSnapshotFailure', { statistic: 'max', ...props });
  }

  public metricClusterCPUUtilization(props?: cw.CommonMetricOptions) {
    return this.baseMetric('CPUUtilization', { statistic: 'max', ...props });
  }

  public metricClusterFreeStorageSpace(props?: cw.CommonMetricOptions) {
    return this.baseMetric('FreeStorageSpace', { statistic: 'min', ...props });
  }

  public metricClusterJVMMemoryPressure(props?: cw.CommonMetricOptions) {
    return this.baseMetric('JVMMemoryPressure', { statistic: 'max', ...props });
  }

  private createSecurityGroup() {
    if (this.securityGroup === undefined) {
      this.securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
        description: `Allows access to the ${this.node.id} service`,
        vpc: this.vpc,
      });

      const vpcOptions = this.vpcOptions as es.CfnDomain.VPCOptionsProperty;
      this.vpcOptions = {
        subnetIds: vpcOptions.subnetIds,
        securityGroupIds: vpcOptions.securityGroupIds
          ? vpcOptions.securityGroupIds.concat(this.securityGroup!.securityGroupId)
          : [this.securityGroup!.securityGroupId],
      };
    }

    return this.securityGroup;
  }

  private baseMetric(metricName: string, props?: cw.CommonMetricOptions) {
    return new cw.Metric({
      dimensions: {
        ClientId: this.config.env.account,
        DomainName: this.domainName,
      },
      namespace: 'AWS/ES',
      ...props,
      metricName,
    });
  }
}
