import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as logs from '@aws-cdk/aws-logs';
import * as ecs from '@aws-cdk/aws-ecs';
import * as elb from '@aws-cdk/aws-elasticloadbalancingv2';
import * as route53 from '@aws-cdk/aws-route53';
import * as r53targets from '@aws-cdk/aws-route53-targets';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { Helper } from './resources/helper';
import { SentryDataStack } from './sentry-data-stack';
import { CommonComputeStack } from './common-compute-stack';

export interface SentryComputeProps {
  dataStack: SentryDataStack;
  commonStack: CommonStack;
  commonCompute: CommonComputeStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class SentryComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: SentryComputeProps) {
    super(scope, config.prefix + 'SentryCompute', { env: config.env });

    this.setupSentry(config, props);
  }

  private setupSentry(config: Config, props: SentryComputeProps) {
    const logGroup = new logs.LogGroup(this, 'SentryLogGroup', {
      logGroupName: 'Sentry',
      retention: logs.RetentionDays.SIX_MONTHS,
    });
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'SentryTask');

    const envVariables = {
      DATABASE_URL:
        'postgresql://' +
        props.dataStack.databaseCluster.secret!.secretValueFromJson('username').toString() +
        ':' +
        props.dataStack.databaseCluster.secret!.secretValueFromJson('password').toString() +
        '@' +
        props.dataStack.databaseCluster.clusterEndpoint.hostname +
        '/sentry',
      SENTRY_REDIS_HOST: props.dataStack.redis.attrPrimaryEndPointAddress,
      SENTRY_SECRET_KEY: props.dataStack.sentrySecretKey.secretValue.toString(),
    };

    taskDefinition
      .addContainer('Sentry', {
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrSentry),
        logging: new ecs.AwsLogDriver({
          streamPrefix: 'Task',
          logGroup,
        }),
        memoryLimitMiB: 1024,
        environment: envVariables,
        command: ['run', 'web'],
      })
      .addPortMappings({ containerPort: 9000 });

    taskDefinition.addContainer('Worker', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrSentry),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 1024,
      environment: envVariables,
      command: ['run', 'worker'],
    });

    taskDefinition.addContainer('Cron', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrSentry),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 256,
      environment: envVariables,
      command: ['run', 'cron'],
    });

    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'SentryService', {
      cluster: props.commonCompute.cluster,
      taskDefinition,
      desiredCount: 1,
      certificate: props.cert,
      openListener: false,
      domainZone: props.commonStack.internalZone,
      domainName: 'sentry-ex.overwolf.wtf',
    });

    service.targetGroup.configureHealthCheck({
      path: '/_health/',
    });

    const sentryPrivateSecurityGroup = new ec2.SecurityGroup(this, 'SentryPrivateLBSG', {
      vpc: props.vpc,
    });

    const sentryPrivateLb = new elb.ApplicationLoadBalancer(this, 'SentryPrivateLB', {
      vpc: props.vpc,
      securityGroup: sentryPrivateSecurityGroup,
    });

    const sentryPrivateListener = sentryPrivateLb.addListener('SentryPrivateListener', {
      port: 443,
      certificates: [props.cert],
      open: false,
    });

    sentryPrivateListener.addTargets('SentryService', {
      port: 80,
      targets: [service.service],
      healthCheck: {
        path: '/_health/',
      },
    });

    new route53.ARecord(this, 'SentryInternalDNS', {
      zone: props.commonStack.internalZone,
      recordName: 'sentry',
      target: route53.RecordTarget.fromAlias(new r53targets.LoadBalancerTarget(sentryPrivateLb)),
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    Helper.addP2PDefaultPortIngress(props.dataStack.databaseCluster.connections, service.service.connections);
    props.dataStack.redis.addIngressPeer(service.service.connections);

    // Ingest
    sentryPrivateSecurityGroup.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(443));
    service.loadBalancer.connections.allowFrom(ec2.Peer.prefixList(config.prefixList), ec2.Port.tcp(443));
  }
}
