import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as rds from '@aws-cdk/aws-rds';
import * as ecr from '@aws-cdk/aws-ecr';
import * as sm from '@aws-cdk/aws-secretsmanager';
import { Config } from './config';
import { ElasticCache } from './resources/elastic-cache';

export interface SentryDataStackProps {
  vpc: ec2.Vpc;
}

export class SentryDataStack extends cdk.Stack {
  databaseCluster: rds.DatabaseCluster;
  ecrSentry: ecr.Repository;
  redis: ElasticCache;
  sentrySecretKey: sm.Secret;

  constructor(scope: cdk.Construct, config: Config, props: SentryDataStackProps) {
    super(scope, config.prefix + 'SentryData', { env: config.env });

    this.ecrSentry = new ecr.Repository(this, 'sentry', {
      repositoryName: 'sentry',
    });

    this.databaseCluster = new rds.DatabaseCluster(this, 'DatabaseCluster', {
      engine: rds.DatabaseClusterEngine.auroraPostgres({
        version: rds.AuroraPostgresEngineVersion.VER_9_6_17,
      }),
      instanceProps: {
        vpc: props.vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.LARGE), // Required by postgres to be this size at least.
      },
      masterUser: { username: 'administrator' },
    });

    this.redis = this.setupElasticCache(config, props.vpc);

    this.sentrySecretKey = new sm.Secret(this, 'SentrySecretKey');
  }

  private setupElasticCache(config: Config, vpc: ec2.Vpc) {
    return new ElasticCache(this, config.prefix + `SentryDataCache`, {
      cacheNodeType: 'cache.t2.small',
      replicasPerNodeGroup: 1,
      replicationGroupDescription: 'Sentry Cache for incoming reports',
      vpc,
    });
  }
}
