package api

import (
	"fmt"
	"net/http"
	"net/http/httputil"
	"net/url"
)

// See https://git.xarth.tv/infosec/cors/blob/master/cors.go for non-preflight only CORs headers.
var corsHeaders = []string{
	"Access-Control-Allow-Origin",
	"Access-Control-Allow-Credentials",
	"Access-Control-Expose-Headers",
}

// NewReverseProxy builds a reverse proxy that sends incoming requests to the targetURL.
// The returned *ReverseProxyTwirpLambda implements http.Handler.
// The serviceName is used to identify the downstream service on logs and errors.
// The targetURL is used to override the incoming request scheme and host, e.g. "https://prod.devsite-rbac.twitch.a2z.com".
func NewReverseProxy(serviceName, targetURL string, stats Statter) *ReverseProxyTwirpHTTP {
	target, err := url.Parse(targetURL)
	if err != nil { // force validation when reading from config
		panic("NewReverseProxy: failed to parse URL " + targetURL + ": " + err.Error())
	}

	return &ReverseProxyTwirpHTTP{
		serviceName: serviceName,
		proxy: &httputil.ReverseProxy{
			// Modify the request into a new request to be sent using Transport.
			Director: func(req *http.Request) {
				req.URL.Scheme = target.Scheme
				req.URL.Host = target.Host

				if _, ok := req.Header["User-Agent"]; !ok {
					req.Header.Set("User-Agent", "Vienna")
				}
			},
			// In case of network error, add the error to the request log and return a 500
			ErrorHandler: func(w http.ResponseWriter, r *http.Request, err error) {
				AddRequestLogKeyErr(r, err)
				w.WriteHeader(500)
				fmt.Fprint(w, `{"code": "internal", "msg": "http: proxy error"}`)
			},
			// Remove CORs headers set by downstream services (e.g. RBAC) to prevent
			// overlap with our CORs policy.
			ModifyResponse: func(r *http.Response) error {
				for _, h := range corsHeaders {
					r.Header.Del(h)
				}
				return nil
			},
		},
		stats: stats,
	}
}

type ReverseProxyTwirpHTTP struct {
	serviceName string
	proxy       *httputil.ReverseProxy
	stats       Statter
}

// ServeHTTP is the method that implements http.Handler, so it can be mounted in a router mux.
// It simply wraps the (*httputil.ReverseProxy).ServeHTTP method, with extra logging.
func (rp *ReverseProxyTwirpHTTP) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	AddRequestLogKeyStr(r, "serviceName", rp.serviceName)

	wr := &writterRecorder{ResponseWriter: w}
	rp.proxy.ServeHTTP(wr, r)

	rp.stats.Inc(fmt.Sprintf("ProxyClient.%s.%s", rp.serviceName, statusGroupXxx(wr.status)), 1) // e.g. "ProxyClient.RBAC.2xx"
}
