#!/usr/bin/env node
import 'source-map-support/register';

import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

import { InterfaceVpcEndpointStack } from './stacks/interface-vpc-endpoint';
import { VPCStack } from './stacks/vpc';
import { Route53Stack } from './stacks/route-53';
import { DeployStack } from './stacks/deploy';
import { EcrRepositoryStack } from './stacks/ecr';
import { ECSViennaStack } from './stacks/ecs-vienna';
import { PROD_ENV, DEV_ENV, GITHUB_VPC_ENDPOINT } from './constants';
import { CertificateStack } from './stacks/certificate';

interface EnvProps {
  prefix: string;
  env: cdk.Environment;
  viennaDomain: string;
  asgInstanceType: ec2.InstanceType;
  asgCapacity: number;
  rbacHost: string;
  rbacVpcEndpoint: string;
  e2IngestHttpLambdaArn: string;
}

const app = new cdk.App();

createStacks(app, {
  prefix: 'Prod',
  env: PROD_ENV,
  viennaDomain: 'vienna.xarth.tv',
  asgInstanceType: new ec2.InstanceType('m5.large'),
  asgCapacity: 3,
  rbacHost: 'prod.devsite-rbac.twitch.a2z.com',
  rbacVpcEndpoint: 'com.amazonaws.vpce.us-west-2.vpce-svc-0ced7ed5454280b71',
  e2IngestHttpLambdaArn:
    'arn:aws:lambda:us-west-2:341681605829:function:TwitchE2IngestHTTPLambda-LambdaFunction-ZNWA75EJF898:live',
});

createStacks(app, {
  prefix: 'Dev',
  env: DEV_ENV,
  viennaDomain: 'vienna-dev.xarth.tv',
  asgInstanceType: new ec2.InstanceType('m5.large'),
  asgCapacity: 1,
  rbacHost: 'staging.devsite-rbac.twitch.a2z.com',
  rbacVpcEndpoint: 'com.amazonaws.vpce.us-west-2.vpce-svc-0a6251c4f6b3b0bf0',
  e2IngestHttpLambdaArn:
    'arn:aws:lambda:us-west-2:597731042549:function:TwitchE2IngestHTTPLambda-LambdaFunction-12NOYMNGIBWTJ:live',
});

export function createStacks(app: cdk.App, props: EnvProps) {
  const { prefix, env } = props;

  const vpcStack = new VPCStack(app, `${prefix}-VPCStack`, { env });
  const vpc = vpcStack.vpc;

  const r53Stack = new Route53Stack(app, `${prefix}-Route53Stack`, {
    env,
    vpc,
    vienna: props.viennaDomain,
  });

  const viennaCertStack = new CertificateStack(app, `${prefix}-CertificateStack`, {
    env,
    domainName: r53Stack.viennaZone.zoneName,
    hostedZone: r53Stack.viennaZone,
  });

  new InterfaceVpcEndpointStack(app, `${prefix}-RBACPrivateLink`, {
    env,
    vpc,
    domainName: props.rbacHost,
    service: { name: props.rbacVpcEndpoint, port: 443 },
  });

  new InterfaceVpcEndpointStack(app, `${prefix}-GHEPrivateLink`, {
    env: env,
    vpc,
    domainName: 'git.xarth.tv',
    service: { name: GITHUB_VPC_ENDPOINT, port: 443 },
  });

  const ecrRepoStack = new EcrRepositoryStack(app, `${prefix}-EcrRepoStack`, {
    env,
    repositoryName: 'vienna',
  });

  const ecsServiceStack = new ECSViennaStack(app, `${prefix}-EcsViennaStack`, {
    env,
    vpc,
    ecrRepo: ecrRepoStack.ecrRepo,
    hostedZone: r53Stack.viennaZone,
    cert: viennaCertStack.cert,

    ec2AsgInstanceType: props.asgInstanceType,
    ec2AsgCapacity: props.asgCapacity,
    containerPort: 8080,
    containerEnv: {
      ENVNAME: prefix.toLowerCase(), // "dev" or "prod"
      PORT: '8080',
      RBAC_HOST: new cdk.StringConcat().join('https://', props.rbacHost),
      E2_INGEST_HTTP_LAMBDA: props.e2IngestHttpLambdaArn,
    },
  });

  const deployStack = new DeployStack(app, `${prefix}-DeployStack`, {
    env,
    vpc,
    ecsService: ecsServiceStack.ecsService,
  });
}
