import * as cdk from '@aws-cdk/core';
import * as certmgr from '@aws-cdk/aws-certificatemanager';
import * as r53 from '@aws-cdk/aws-route53';

export interface CertificateStackProps extends cdk.StackProps {
  domainName: string;
  hostedZone: r53.IHostedZone;
}

// Certificate to allow SSL/HTTPS requests using a domain hosted zone.
// The hosted zone needs to be delegated to this AWS account, for example,
// the vienna.xarth.tv zone was delegated to the vienna-prod account using
// this JIRA ticket: https://jira.twitch.com/browse/TWIT-3530 (and some Slack DMs).
// The hosted zone needs to be public, so it is reachable from outside of the VPN.
// The Certificate will be validated with DNS, which means that a new CNAME record
// is created on route53 and then Certificate Manager checks if it is reachable.
export class CertificateStack extends cdk.Stack {
  public readonly cert: certmgr.ICertificate;

  constructor(scope: cdk.Construct, id: string, props: CertificateStackProps) {
    super(scope, id, props);

    this.cert = new certmgr.Certificate(this, 'Cert', {
      domainName: props.domainName,
      validation: certmgr.CertificateValidation.fromDns(props.hostedZone),
    });
  }
}
