import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as codebuild from '@aws-cdk/aws-codebuild';
import * as codepipeline from '@aws-cdk/aws-codepipeline';
import * as codepipelineActions from '@aws-cdk/aws-codepipeline-actions';
import { VIENNA_REPO } from '../constants';

export interface DeployStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
  ecsService: ecs.Ec2Service;
}

export class DeployStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: DeployStackProps) {
    super(scope, id, props);

    const ecrRepo = ecr.Repository.fromRepositoryName(this, 'ECRRepo', 'vienna');

    const githubSource = codebuild.Source.gitHubEnterprise({
      httpsCloneUrl: VIENNA_REPO,
      branchOrRef: 'master',
    });

    const viennaBuildProject = new codebuild.Project(this, 'vienna-build-project', {
      source: githubSource,
      description: 'Get Vienna Source from GHE, build, and push image to ECR',
      projectName: 'vienna-build-from-pipeline',
      vpc: props.vpc,
      environment: {
        // to use Docker commands in build process
        privileged: true,
        buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
      },
      environmentVariables: {
        //ECR Repo
        REPOSITORY_URI: {
          value: `${props.env?.account}.dkr.ecr.${props.env?.region}.amazonaws.com/vienna`,
        },
        ACCOUNT_REGION: { value: props.env?.region },
      },
      buildSpec: codebuild.BuildSpec.fromObject({
        version: '0.2',
        phases: {
          build: {
            pre_build: {
              commands: [
                'echo Logging into ECR...',
                'aws --version',
                'aws ecr get-login-password --region $ACCOUNT_REGION | docker login --username AWS --password-stdin $REPOSITORY_URI',
              ],
            },
            build: {
              commands: [
                'echo Building Docker image...',
                'docker build -t vienna .',
                'docker tag vienna $REPOSITORY_URI',
              ],
            },
            post_build: {
              commands: ['echo Pushing Docker image to ECR...', 'docker push $REPOSITORY_URI'],
            },
          },
        },
      }),
    });
    ecrRepo.grantPullPush(viennaBuildProject.grantPrincipal);

    const githubSecurityGroup = new ec2.SecurityGroup(this, 'vienna-fetch-source-ghe-sg', {
      vpc: props.vpc,
      securityGroupName: 'Github Enterprise 2',
      description: 'Github Enterprise VPC Access',
    });
    githubSecurityGroup.connections.allowFrom(
      viennaBuildProject.connections,
      ec2.Port.tcp(443),
      'Allow Access to GHE',
    );
    githubSecurityGroup.connections.allowTo(
      viennaBuildProject.connections,
      ec2.Port.tcp(443),
      'Allow Access to GHE',
    );

    // converts imageDetail.json returned by ECR Source to imagedefinitions.json expected by Deploy stage
    // imagedefinitions.json must have 2 fields: name and imageUri
    const viennaBuildArtifactProject = new codebuild.PipelineProject(
      this,
      'vienna-build-artifact-project',
      {
        description: 'Convert ECR artifact into artifact suitable for Deploy stage',
        projectName: 'vienna-build-artifact',
        vpc: props.vpc,
        environment: {
          buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
        },
        environmentVariables: {
          CONTAINER_NAME: { value: 'TaskContainer' },
        },
        buildSpec: codebuild.BuildSpec.fromObject({
          version: '0.2',
          phases: {
            build: {
              commands: [
                'IMAGE_URI=$(cat imageDetail.json | python -c "import sys, json; print(json.load(sys.stdin)[\\"ImageURI\\"].split(\\"@\\")[0]) ")',
                'IMAGE_TAG=$(cat imageDetail.json | python -c "import sys, json; print(json.load(sys.stdin)[\\"ImageTags\\"][0]) ")',
                'echo $IMAGE_URI:$IMAGE_TAG',
                'echo Writing image defitions file...',
                'printf "[{\\"name\\":\\"%s\\",\\"imageUri\\":\\"%s\\"}]" $CONTAINER_NAME $IMAGE_URI:$IMAGE_TAG > imagedefinitions.json',
                'cat imagedefinitions.json',
              ],
            },
          },
          artifacts: {
            files: ['**/*'],
          },
        }),
      },
    );

    const viennaPipeline = new codepipeline.Pipeline(this, 'Vienna-Pipeline', {
      pipelineName: 'vienna-deploy',
    });

    const viennaSourceArtifact = new codepipeline.Artifact();
    const viennaBuildArtifact = new codepipeline.Artifact();

    viennaPipeline.addStage({
      stageName: 'Source',
      actions: [
        new codepipelineActions.EcrSourceAction({
          actionName: 'vienna-ecr-source',
          repository: ecrRepo,
          imageTag: 'latest',
          output: viennaSourceArtifact,
        }),
      ],
    });

    viennaPipeline.addStage({
      stageName: 'BuildArtifact',
      actions: [
        new codepipelineActions.CodeBuildAction({
          actionName: 'vienna-build-artifact',
          project: viennaBuildArtifactProject,
          input: viennaSourceArtifact,
          outputs: [viennaBuildArtifact],
        }),
      ],
    });

    viennaPipeline.addStage({
      stageName: 'Deploy',
      actions: [
        new codepipelineActions.EcsDeployAction({
          actionName: 'vienna-ecs-deploy',
          service: props.ecsService,
          input: viennaBuildArtifact,
        }),
      ],
    });
  }
}
