import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as elbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import * as certmgr from '@aws-cdk/aws-certificatemanager';
import * as r53 from '@aws-cdk/aws-route53';
import * as r53targets from '@aws-cdk/aws-route53-targets';
import * as cdk from '@aws-cdk/core';
import { ManagedPolicy } from '@aws-cdk/aws-iam';
import { Tag, TagManager } from '@aws-cdk/core';

const WPA2_APLL = 'pl-f8a64391'; // predefined prefix list to allow access from WPA2/AmazonVPN

interface ECSViennaStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
  ec2AsgInstanceType: ec2.InstanceType;
  ec2AsgCapacity: number;
  ecrRepo: ecr.IRepository;
  hostedZone: r53.IHostedZone;
  cert: certmgr.ICertificate;
  containerPort: number; // tcp port to access the service container
  containerEnv: {
    [key: string]: string; // --env variables passed to the container, used on api/config.go
  };
}

export class ECSViennaStack extends cdk.Stack {
  public readonly ecsService: ecs.Ec2Service;

  constructor(scope: cdk.Construct, id: string, props: ECSViennaStackProps) {
    super(scope, id, props);

    // Cluster
    const cluster = new ecs.Cluster(this, `Cluster`, {
      vpc: props.vpc,
      containerInsights: true,
    });

    // ASG
    const asg = cluster.addCapacity('ASG', {
      instanceType: props.ec2AsgInstanceType,
      minCapacity: props.ec2AsgCapacity,
      maxCapacity: props.ec2AsgCapacity,
    });
    asg.role.addManagedPolicy(
      ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
    );

    // Logs
    let logGroup = new logs.LogGroup(this, 'ViennaBackendLogGroup', {
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    // Task Definition
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDefinition', {});
    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'], // custom metrics from TwitchTelemetry
        resources: ['*'],
      }),
    );
    taskDefinition.taskRole.addManagedPolicy(
      ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaRole'),
    );

    const container = taskDefinition.addContainer('TaskContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.ecrRepo, 'latest'), // Requires props.ecrRepo to exist and have a registered docker image
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'vienna',
        logGroup: logGroup,
      }),
      memoryReservationMiB: 1024, // 1 GB
      // maps to --env option to docker run
      environment: props.containerEnv,
    });
    container.addPortMappings({ containerPort: props.containerPort });

    // Service
    this.ecsService = new ecs.Ec2Service(this, 'TaskService', {
      serviceName: 'Vienna',
      cluster: cluster,
      taskDefinition,
      assignPublicIp: false, // accessed through a LB
      desiredCount: props.ec2AsgCapacity,
    });

    // Load Balancer with access from WPA2/AmazonVPN
    const sgWpa2 = new ec2.SecurityGroup(this, 'SecurityGroupWpa2', { vpc: props.vpc });
    sgWpa2.addIngressRule(
      ec2.Peer.prefixList(WPA2_APLL),
      ec2.Port.tcp(443),
      'Amazon VPN/WPA2 https',
    );

    // Load Balancer with access from WPA2/AmazonVPN
    const albWpa2 = new elbv2.ApplicationLoadBalancer(this, 'AlbWpa2', {
      vpc: props.vpc,
      internetFacing: true,
    });
    albWpa2.addRedirect({
      sourceProtocol: elbv2.ApplicationProtocol.HTTP,
      sourcePort: 80,
      targetProtocol: elbv2.ApplicationProtocol.HTTPS,
      targetPort: 443,
    });
    const albWpa2Listener = albWpa2.addListener('ListenerHttps', {
      port: 443, // default https port
      certificates: [props.cert], // for https
      open: false,
    });
    albWpa2Listener.addTargets('TargetGroupVienna', {
      port: props.containerPort,
      targets: [this.ecsService],
      healthCheck: { path: '/health' },
    });
    albWpa2Listener.connections.allowFrom(
      ec2.Peer.prefixList(WPA2_APLL),
      ec2.Port.tcp(443), // WPA2/AmazonVPN access
    );

    new r53.ARecord(this, 'Route53Target', {
      zone: props.hostedZone,
      target: r53.AddressRecordTarget.fromAlias(new r53targets.LoadBalancerTarget(albWpa2)),
    });
  }
}
