package main

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/gorilla/mux"
	"github.com/rs/zerolog"
	"github.com/rs/zerolog/log"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/infosec/cors"
	"code.justin.tv/lifecycle/vienna/api"
)

func main() {
	conf := api.ParseConfig() // read environment variables and flags
	if conf.IsLocal() {
		log.Logger = log.Output(zerolog.ConsoleWriter{Out: os.Stderr}) // Pretty print logs
	}

	if len(os.Args) >= 2 && os.Args[1] == "-h" {
		conf.PrintUsage()
		return
	} else {
		log.Info().Msgf("Start with conf: %+v", conf)
	}

	var stats api.Statter
	if conf.IsLocal() {
		stats = &api.NullStatter{} // no stats
	} else {
		stats = api.NewTelemetryStatter(identifier.ProcessIdentifier{
			Service: "Vienna",
			Stage:   conf.ENVNAME,
			Region:  "us-west-2",
		})
	}

	// Middleware
	r := mux.NewRouter()
	r.Use(api.LoggingMiddleware(log.Logger, stats))
	policy := cors.Policy{
		AllowedOrigins: cors.Origins("http://localhost:3000"), // Must match the devServer config in webpack.config.js
		AllowHeaders:   cors.Headers("Authorization", "Content-Type"),
		MaxAge:         cors.MaxAge(10 * time.Minute),
	}
	r.Use(policy.MustMiddleware)
	r.Use(cors.BlockOnOptions)

	// API dependencies and Routes
	a := api.NewApi(conf, stats)
	a.SetupRoutes(r)

	// HTTP Server
	srv := &http.Server{
		Addr:         fmt.Sprintf(":%d", conf.PORT),
		WriteTimeout: 15 * time.Second,
		ReadTimeout:  15 * time.Second,
		IdleTimeout:  60 * time.Second,
		Handler:      r,
	}
	go func() {
		if err := srv.ListenAndServe(); err != nil {
			log.Fatal().Err(err).Msg("ListenAndServe startup failed")
		}
	}()
	log.Info().Msgf("listening on port %d", conf.PORT)

	// Block and wait until quit via SIGINT (Ctrl+C),
	// or SIGTERM (Ctrl+/, or when ECS stops a task.
	c := make(chan os.Signal, 1)
	signal.Notify(c, os.Interrupt, syscall.SIGTERM, syscall.SIGUSR2)
	<-c

	// Graceful shutdown
	// Create a deadline to wait for existing connections to finish
	gracefulWait := 15 * time.Second
	ctx, cancel := context.WithTimeout(context.Background(), gracefulWait)
	defer cancel()
	_ = srv.Shutdown(ctx) // Doesn't block if no connections, but will otherwise wait until the timeout deadline.
	log.Info().Msg("shutting down")
	os.Exit(0)
}
