import React, { useState, useCallback, useEffect } from 'react';
import { ModalHeader, ModalBody, Layout, CoreText, Display } from 'twitch-core-ui';
import { Modal, ModalId } from 'tachyon-modal';
import {
  ListActionHistoriesRequest,
  ActionHistory,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { ActionHistoryEntityType } from '~features/action-history/models/entity-type';
import { DateTime } from 'luxon';
import { RBACActionHistoryClient } from '~core/vienna';

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  active: boolean;
  entityId?: string;
  companyId?: string;
  entityType?: ActionHistoryEntityType;
}

type Props = PublicProps;

const ACTION_ITEM_LIMIT = 25;

export const ActionHistoryModal: React.FC<Props> = (props) => {
  const [historyItems, setHistoryItems] = useState<ActionHistory[]>([]);

  async function getActivityItems() {
    if (!props.active) {
      return;
    }

    const activityLog = await RBACActionHistoryClient.listActionHistories(
      new ListActionHistoriesRequest({
        entityId: props.entityId,
        companyId: props.companyId,
        entityType: props.entityType,
        limit: ACTION_ITEM_LIMIT,
      }),
    );

    const items = activityLog.actionHistories;
    setHistoryItems(items);
  }

  const activityItemsCallback = useCallback(getActivityItems, [
    props.active,
    props.entityId,
    props.companyId,
    props.entityType,
  ]);

  useEffect(() => {
    activityItemsCallback();
  }, [activityItemsCallback]);

  return (
    <Modal id={props.id}>
      <ModalHeader
        title="Action History"
        closeButton={{ onClick: props.closeModal, 'aria-label': 'Close' }}
      />
      <ModalBody>
        <Layout>
          {historyItems.map((item, index) => {
            const date = DateTime.fromISO(item.createdAt).toLocaleString(DateTime.DATETIME_MED);
            return (
              <Layout padding={{ y: 2 }} key={index} display={Display.Flex}>
                <Layout flexGrow={1} flexShrink={1} margin={{ right: 2 }}>
                  <CoreText>{date}</CoreText>
                </Layout>
                <Layout>
                  <CoreText>{item.action}</CoreText>
                </Layout>
              </Layout>
            );
          })}
        </Layout>
      </ModalBody>
    </Modal>
  );
};
