import React, { useState } from 'react';
import {
  Layout,
  Input,
  Position,
  InputType,
  InjectStyledLayout,
  Interactable,
  CoreText,
  Balloon,
  SVGAsset,
} from 'twitch-core-ui';
import { useCombobox } from 'downshift';

export interface PublicProps<T> {
  items: T[];
  searchPlaceholder: string;
  onSearchChange: (term: string) => void;
  onSelect: (item: T) => void;
  itemToString: (item?: T) => string;
  child?: React.ComponentType<AutocompleteChildProps<T>>;
  initialSelectedItem?: T;
}

export interface AutocompleteChildProps<T> {
  selectedItem?: T;
  reset?: () => void;
}

type Props<T> = PublicProps<T>;

export const Autocomplete = <T extends { id: string }>(props: Props<T>) => {
  const initialSearchTerm = props.itemToString(props.initialSelectedItem);
  const [searchTerm, setSearchTerm] = useState(initialSearchTerm);
  const {
    selectedItem,
    highlightedIndex,
    isOpen,
    getComboboxProps,
    getInputProps,
    getItemProps,
    getMenuProps,
    openMenu,
    closeMenu,
    reset,
  } = useCombobox({
    items: props.items,
    initialInputValue: searchTerm,
    initialSelectedItem: props.initialSelectedItem,
    onInputValueChange: (ev) => {
      if (ev.inputValue !== undefined) {
        props.onSearchChange(ev.inputValue);
        setSearchTerm(ev.inputValue);
      }
    },
    onSelectedItemChange: (ev) => {
      if (ev.selectedItem) {
        props.onSelect(ev.selectedItem);
        closeMenu();
      }
    },
    itemToString: props.itemToString,
  });

  return (
    <>
      <Layout position={Position.Relative}>
        <div {...getComboboxProps()}>
          <Input
            type={InputType.Text}
            {...getInputProps({ refKey: 'refDelegate', value: searchTerm })}
            icon={SVGAsset.NavSearch}
            placeholder={props.searchPlaceholder}
            onFocus={openMenu}
          />
        </div>
        <Balloon show={isOpen} elevation={2} tail={false}>
          <ul {...getMenuProps()}>
            {props.items.map((item, index) => (
              <InjectStyledLayout key={`${item.id}${index}`}>
                <li {...getItemProps({ item, index })}>
                  <Interactable hover={highlightedIndex === index}>
                    <Layout padding={{ x: 2, y: 1 }}>{props.itemToString(item)}</Layout>
                  </Interactable>
                </li>
              </InjectStyledLayout>
            ))}
            {searchTerm !== '' && props.items.length === 0 && (
              <Layout padding={{ x: 2, y: 1 }}>
                <CoreText>No Search Results</CoreText>
              </Layout>
            )}
          </ul>
        </Balloon>
      </Layout>
      {props.child && <props.child selectedItem={selectedItem} reset={reset} />}
    </>
  );
};
