import React, { useState, useEffect } from 'react';
import { DebounceTimer } from '~core/utils/debounce-timer';
import { Company, SearchCompaniesRequest } from '~core/clients/rbac/code.justin.tv.devrel';
import { RBACClient } from '~core/vienna';
import { Autocomplete } from '~features/autocomplete/autocomplete';

const debounceTimer = new DebounceTimer(200);

export interface PublicProps {
  onCompanySelect: (company?: Company) => void;
  selectedCompany?: Company;
}

type Props = PublicProps;

export const CompanyAutocomplete: React.FC<Props> = (props) => {
  const [companies, setCompanies] = useState<Company[]>([]);

  async function searchCompanies(searchTerm: string) {
    const searchResponse = await RBACClient.searchCompanies(
      new SearchCompaniesRequest({ query: searchTerm }),
    );
    setCompanies(searchResponse.companies);
  }

  function onSearchChange(term: string) {
    if (!term.length) {
      setCompanies([]);
      props.onCompanySelect(undefined);
      return;
    }

    debounceTimer.invoke(() => {
      searchCompanies(term);
    });
  }

  useEffect(() => {
    if (props.selectedCompany) {
      searchCompanies(props.selectedCompany.companyName);
    }
  }, [props.selectedCompany]);

  return (
    <Autocomplete<Company>
      items={companies}
      searchPlaceholder="Search for an organization"
      onSearchChange={onSearchChange}
      onSelect={props.onCompanySelect}
      itemToString={(item?: Company) => item?.companyName || ''}
      initialSelectedItem={props.selectedCompany}
    />
  );
};
