import React from 'react';
import {
  Layout,
  Display,
  JustifyContent,
  AlignItems,
  CoreText,
  TableHeader,
  TableRow,
  TableHeading,
  Table,
  TableBody,
  LoadingSpinner,
} from 'twitch-core-ui';
import { ListingHeader } from '~features/listing-header';
import { PaginationProps, CursorPaginationProps } from '~features/hooks/use-pagination';

export interface EntityProps<T> {
  item: T;
}

export interface PublicProps<T, P> {
  columns: string[];
  items: T[];
  loading?: boolean;
  zeroStateMessage: string;
  rowType: React.ComponentType<EntityProps<T> & P>;
  rowProps: P;
  header: {
    searchLabel?: string;
    searchTerm?: string;
    onSearchUpdate?: (term: string) => void;
    searchPlaceholder?: string;
    additionalFields?: React.ReactChild[];
  };
  cursorPagination?: CursorPaginationProps;
  pagination?: PaginationProps;
}

type Props<T, P> = PublicProps<T, P>;

export const EntityList = <T extends {}, P = {}>(props: React.PropsWithChildren<Props<T, P>>) => {
  return (
    <Layout fullWidth>
      <ListingHeader
        pagination={props.pagination}
        cursorPagination={props.cursorPagination}
        searchPlaceholder={props.header.searchPlaceholder}
        searchLabel={props.header.searchLabel}
        searchTerm={props.header.searchTerm}
        onSearchChange={props.header.onSearchUpdate}
        additionalFields={props.header.additionalFields}
      />
      {props.items.length === 0 && !props.loading && (
        <Layout
          justifyContent={JustifyContent.Center}
          alignItems={AlignItems.Center}
          fullHeight
          fullWidth
          display={Display.Flex}
        >
          <CoreText>{props.zeroStateMessage}</CoreText>
        </Layout>
      )}

      {props.items.length > 0 && (
        <Table alternateRows>
          <TableHeader>
            <TableRow>
              {props.columns.map((column, index) => (
                <TableHeading label={column} key={index} />
              ))}
            </TableRow>
          </TableHeader>

          {!props.loading && (
            <TableBody>
              {props.items.map((item, index) => (
                <props.rowType {...props.rowProps} item={item} key={index} />
              ))}
            </TableBody>
          )}
        </Table>
      )}
      {props.loading && (
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          alignItems={AlignItems.Center}
          fullWidth
          fullHeight
        >
          <LoadingSpinner />
        </Layout>
      )}
    </Layout>
  );
};
