import React, { useState, useEffect } from 'react';
import { useLazyQuery } from '@apollo/react-hooks';
import searchGameQuery from './queries/search-game-query.gql';
import {
  SearchGamesVariables,
  SearchGames,
  SearchGames_searchFor_games_items,
} from '~core/graphql/twitch-gql-schema';
import { DebounceTimer } from '~core/utils/debounce-timer';
import { Game } from '~features/game-autocomplete/models/game';
import { Autocomplete, AutocompleteChildProps } from '~features/autocomplete/autocomplete';

const debounceTimer = new DebounceTimer(200);

export interface PublicProps {
  onGameSelect: (game?: Game) => void;
  child?: React.ComponentType<AutocompleteChildProps<SearchGames_searchFor_games_items>>;
}

type Props = PublicProps;

export const GameAutocomplete: React.FC<Props> = (props) => {
  const [games, setGames] = useState<SearchGames_searchFor_games_items[]>([]);

  const [loadGames, { data }] = useLazyQuery<SearchGames, SearchGamesVariables>(searchGameQuery);

  function onSearchChange(term: string) {
    if (!term.length) {
      setGames([]);
      props.onGameSelect(undefined);
      return;
    }

    debounceTimer.invoke(() => {
      loadGames({ variables: { query: term } });
    });
  }

  useEffect(() => {
    if (data?.searchFor?.games?.items) {
      setGames(data.searchFor.games.items);
    }
  }, [data, setGames, data?.searchFor?.games?.items]);

  return (
    <Autocomplete<SearchGames_searchFor_games_items>
      items={games}
      searchPlaceholder="Search for a game"
      onSearchChange={onSearchChange}
      onSelect={props.onGameSelect}
      itemToString={(item?: SearchGames_searchFor_games_items) => item?.displayName || ''}
      child={props.child}
    />
  );
};
