import React, { useState, useCallback, useEffect } from 'react';
import {
  GameApplication,
  Id,
  ListCompaniesRequest,
  Company,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { TableRow, TableCell, CoreLink, CoreText } from 'twitch-core-ui';
import { RBACClient } from '~core/vienna';
import { DateTime } from 'luxon';
import gameInfoQuery from './queries/get-game-query.gql';
import { useQuery } from '@apollo/react-hooks';
import { getGameInfo, getGameInfoVariables } from '~core/graphql/twitch-gql-schema';
import { EntityProps } from '~features/entity-list';

export interface PublicProps {}

type Props = PublicProps & EntityProps<GameApplication>;

export const GameReviewListItem: React.FC<Props> = (props) => {
  const [companyName, setCompanyName] = useState('');
  const [owningCompany, setOwningCompany] = useState<Company>();
  const { data } = useQuery<getGameInfo, getGameInfoVariables>(gameInfoQuery, {
    variables: { query: props.item.gameName },
    skip: false,
  });

  async function getCompany() {
    try {
      const result = await RBACClient.getCompany(new Id({ id: props.item.companyId }));
      setCompanyName(result.companyName);
    } catch {}
  }
  const getCompanyCallback = useCallback(getCompany, [props.item.companyId]);

  async function getGameOwnership() {
    try {
      const result = await RBACClient.listCompanies(
        new ListCompaniesRequest({
          ownsGameId: props.item.gameId,
          limit: 1,
          offset: 0,
        }),
      );

      if (result.companies?.length) {
        setOwningCompany(result.companies[0]);
      }
    } catch {}
  }
  const getGameOwnershipCallback = useCallback(getGameOwnership, [props.item.gameId]);

  useEffect(() => {
    getCompanyCallback();
    getGameOwnershipCallback();
  }, [getCompanyCallback, getGameOwnershipCallback]);

  const giantBombId =
    data?.searchFor?.games?.items?.length && data.searchFor.games.items[0].giantBombID;

  return (
    <TableRow>
      <TableCell>
        <CoreLink linkTo={`/games/review/${props.item.id}`}>
          {props.item.gameName || 'Unknown'}
        </CoreLink>
      </TableCell>
      <TableCell>{props.item.gameId}</TableCell>
      <TableCell>
        <CoreLink linkTo={`/organizations/${props.item.companyId}`}>{companyName}</CoreLink>
      </TableCell>
      <TableCell>
        <CoreLink linkTo={`/organizations/${owningCompany?.id}`}>
          {owningCompany?.companyName}
        </CoreLink>
      </TableCell>
      <TableCell>
        {DateTime.fromISO(props.item.createdAt).toLocaleString(DateTime.DATETIME_MED)}
      </TableCell>
      <TableCell>N/A</TableCell>
      <TableCell>
        {!giantBombId && <CoreText>N/A</CoreText>}
        {giantBombId && (
          <CoreLink targetBlank linkTo={`https://www.giantbomb.com/games/3030-${giantBombId}`}>
            Giant Bomb
          </CoreLink>
        )}
      </TableCell>
    </TableRow>
  );
};
