import { useLocation, useHistory } from 'react-router';
import { parse, stringify } from 'query-string';
import { useState, useCallback } from 'react';

export interface PaginationProps {
  currentPage: number;
  totalPages: number;
  onNextPage: () => void;
  onPreviousPage: () => void;
  onGoToPage: (index: number) => void;
}

export interface CursorPaginationProps {
  hasNext: boolean;
  hasPrevious: boolean;
  totalPages: number;
  currentPage: number;
  onNextPage: () => void;
  onPreviousPage: () => void;
}

export interface Pagination {
  page: number;
}

export function usePagination() {
  const { search, pathname } = useLocation();
  const history = useHistory();

  const queryString = search;
  const queryParams = parse(queryString, { parseNumbers: true });
  const paramPage = (queryParams.page as number) || 1;

  const [page, setPage] = useState<number>(paramPage);

  function previousPage() {
    if (paramPage <= 1) {
      return;
    }
    goToPage(paramPage - 1);
  }

  function nextPage() {
    goToPage(paramPage + 1);
  }

  function goToPage(newPage: number) {
    if (newPage < 1) {
      newPage = 1;
    }
    updateURLCallback(newPage);
  }

  function updateURL(page: number) {
    queryParams.page = page;
    setPage(page);

    history.push({
      pathname: pathname,
      search: stringify(queryParams),
    });
  }

  const updateURLCallback = useCallback(updateURL, [history, pathname, queryParams]);

  const goToPageCallback = useCallback(goToPage, [updateURLCallback]);

  return {
    page,
    previousPage,
    nextPage,
    goToPage: goToPageCallback,
  };
}
