import React, { useEffect, useState } from 'react';
import { ModalId, Modal } from 'tachyon-modal';
import { DeveloperApplication, Empty } from '~core/clients/rbac/code.justin.tv.devrel';
import {
  ModalHeader,
  ModalBody,
  Layout,
  Title,
  TitleSize,
  Radio,
  ModalFooter,
} from 'twitch-core-ui';
import * as yup from 'yup';
import { useFormik } from 'formik';
import { InfoField } from 'src/layout/info-field';
import { RBACClient } from '~core/vienna';

export interface PublicProps {
  modalId: ModalId;
  closeModal: () => void;
  onClose: (userId: string, role: string) => void;
  developer: DeveloperApplication;
  active: boolean;
}

type Props = PublicProps;

export interface DeveloperApplicationForm {
  role: string;
}

const formValidation = yup.object<DeveloperApplicationForm>({
  role: yup.string().required(),
});

export const DeveloperApplicationModal: React.FC<Props> = (props) => {
  const [roles, setRoles] = useState<string[]>([]);
  const formik = useFormik<DeveloperApplicationForm>({
    onSubmit: submitForm,
    initialValues: {
      role: '',
    },
    validationSchema: formValidation,
    validateOnMount: true,
  });

  function submitForm(values: DeveloperApplicationForm) {
    props.onClose(props.developer.id, values.role);
  }

  function completeForm(e: React.MouseEvent<HTMLButtonElement>) {
    e.preventDefault();
    formik.submitForm();
  }

  useEffect(() => {
    (async () => {
      try {
        const rolesResponse = await RBACClient.listAllRoles(new Empty());
        setRoles(rolesResponse.roles);
      } catch {}
    })();
  }, [props.active]);

  return (
    <Modal id={props.modalId}>
      <ModalHeader
        title="Add User"
        closeButton={{ 'aria-label': 'Close', onClick: props.closeModal }}
      />
      <form>
        <ModalBody>
          <InfoField
            title="Developer"
            value={`${props.developer.firstName} ${props.developer.lastName}`}
          ></InfoField>
          <Layout>
            <Layout margin={{ bottom: 0.5 }}>
              <Title size={TitleSize.ExtraSmall}>Select Role:</Title>
            </Layout>
            {roles.map((role, index) => (
              <Radio
                value={role}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                checked={formik.values.role === role}
                name="role"
                label={role}
                key={index}
              />
            ))}
          </Layout>
        </ModalBody>
        <ModalFooter
          primaryButtonProps={{
            onClick: completeForm,
            children: 'Add Member',
            disabled: !formik.isValid,
          }}
          secondaryButtonProps={{
            onClick: props.closeModal,
            children: 'Cancel',
            disabled: !formik.isValid,
          }}
        />
      </form>
    </Modal>
  );
};
