import React from 'react';
import {
  TableRow,
  TableCell,
  Display,
  Layout,
  JustifyContent,
  Button,
  ButtonType,
  Interactable,
} from 'twitch-core-ui';
import { Membership } from '~core/clients/rbac/code.justin.tv.devrel';
import { useModal } from 'tachyon-modal';
import { ConfirmationDialog } from '~features/confirmation-dialog/confirmation-dialog';
import { ManageUserModal } from '~features/manage-user-modal';
import { UserInformation } from '~features/organizations/models/user-information';
import { ActionHistoryModal } from '~features/action-history-modal';

export interface PublicProps {
  member: Membership;
  onRemoveUser: (userId: string) => void;
  onUpdateUser: (user: UserInformation) => void;
}

type Props = PublicProps;

export const OrganizationUserItem: React.FC<Props> = (props) => {
  function onRemoveUser(confirmation: boolean) {
    if (confirmation) {
      props.onRemoveUser(props.member.twitchId);
    }
  }

  function onUpdateUser(user: UserInformation) {
    props.onUpdateUser(user);
  }

  const { openModal, id, active, closeModal } = useModal();
  const {
    openModal: openActionModal,
    id: actionModalId,
    active: actionModalActive,
    closeModal: actionModalClose,
  } = useModal();
  const {
    openModal: openManageUserModal,
    active: manageUserActive,
    id: manageUserId,
    closeModal: closeManageUser,
  } = useModal();

  return (
    <>
      <TableRow>
        <TableCell>{props.member.twitchId}</TableCell>

        <TableCell>
          <Interactable onClick={openManageUserModal} hover={false} blurAfterClick>
            {props.member.firstName} {props.member.lastName}
          </Interactable>
        </TableCell>
        <TableCell>{props.member.devTitle}</TableCell>
        <TableCell>{props.member.role}</TableCell>
        <TableCell>{props.member.devEmail}</TableCell>
        <TableCell>{props.member.createdAt}</TableCell>
        <TableCell>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Between} padding={2}>
            <Button onClick={openActionModal} type={ButtonType.Secondary}>
              Activity Log
            </Button>
            <Button type={ButtonType.Alert} onClick={openModal}>
              Delete
            </Button>
          </Layout>
        </TableCell>
      </TableRow>
      <ConfirmationDialog
        id={id}
        closeModal={closeModal}
        active={active}
        text={`Are you sure you want to remove ${props.member.firstName} ${props.member.lastName}?`}
        onClose={onRemoveUser}
      />
      <ManageUserModal
        id={manageUserId}
        active={manageUserActive}
        closeModal={closeManageUser}
        user={{
          firstName: props.member.firstName,
          lastName: props.member.lastName,
          title: props.member.devTitle,
          email: props.member.devEmail,
          role: props.member.role,
          twitchUserId: props.member.twitchId,
        }}
        onClose={onUpdateUser}
      />
      <ActionHistoryModal
        active={actionModalActive}
        id={actionModalId}
        entityId={props.member.twitchId}
        closeModal={actionModalClose}
        companyId={props.member.companyId}
      />
    </>
  );
};
