import React, { useState } from 'react';
import { Game } from '~core/clients/rbac/code.justin.tv.devrel';
import {
  TableRow,
  TableCell,
  Display,
  Layout,
  JustifyContent,
  Button,
  ButtonType,
  DropDownMenu,
  DropDownMenuItem,
  StyledLayout,
  Position,
} from 'twitch-core-ui';
import { ClickOutDetector } from '~core/components/click-out-detector';
import { krakenClient } from '~core/vienna';
import { toast } from 'react-toastify';
import { useModal } from 'tachyon-modal';
import { ConfirmationDialog } from '~features/confirmation-dialog/confirmation-dialog';

export interface PublicProps {
  game: Game;
  onActivityModal: (gameId: string) => void;
  onRemoveGame: (gameId: number) => void;
}

type Props = PublicProps;

export const OrganizationGamesItem: React.FC<Props> = (props) => {
  const [menuOpen, setMenuOpen] = useState(false);
  const {
    openModal: removeGameOpenModal,
    id: removeGameId,
    active: removeGameActive,
    closeModal: removeGameCloseModal,
  } = useModal();

  async function onInsightsReportCSV() {
    const url = `/v5/piper/report/games/${props.game.id}/overview_v2/latest`;
    const response = await krakenClient.get(url);

    if (response.status === 404) {
      toast('There is no report for this game', { type: 'error' });
      setMenuOpen(false);
      return;
    }

    if (response.errorMessage) {
      toast(`Unable to Download CSV Report: ${response.errorMessage}`, { type: 'error' });
      setMenuOpen(false);

      return;
    }

    const reportUrl = response.json;
    window.location.href = reportUrl;
    setMenuOpen(false);
  }

  async function onQuarterlyReport() {
    const today = new Date();
    const { year, quarter } = lastQuarterWithAvailableOnePagerReport(today);

    const url = `/v5/piper/onepager/games/${props.game.id}/download?year=${year}&quarter=${quarter}`;
    const response = await krakenClient.get(url);

    if (response.status === 404) {
      toast(`There is no report for Q${quarter} ${year}`, { type: 'error' });
      setMenuOpen(false);

      return;
    }

    if (response.errorMessage) {
      toast(`Unable to Download CSV Report: ${response.errorMessage}`, { type: 'error' });
      setMenuOpen(false);

      return;
    }
    const reportUrl = response.json;
    window.location.href = reportUrl;
    setMenuOpen(false);
  }

  function onActivityModal() {
    props.onActivityModal(props.game.id.toString());
  }

  function onRemove(confirmation: boolean) {
    if (confirmation) {
      props.onRemoveGame(props.game.id);
    }
  }

  return (
    <>
      <TableRow>
        <TableCell>{props.game.id}</TableCell>
        <TableCell>{props.game.name}</TableCell>
        <TableCell>{props.game.giantbombId}</TableCell>
        <TableCell>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Around}>
            <StyledLayout position={Position.Relative}>
              <ClickOutDetector onClickOut={() => setMenuOpen(false)}>
                <Button
                  dropdown
                  type={ButtonType.Secondary}
                  onClick={() => setMenuOpen((open) => !open)}
                  blurAfterClick
                >
                  Actions
                </Button>
                <DropDownMenu show={menuOpen}>
                  <DropDownMenuItem onClick={onActivityModal}>Activity Log</DropDownMenuItem>
                  <DropDownMenuItem onClick={onInsightsReportCSV}>
                    Insights CSV Report
                  </DropDownMenuItem>
                  <DropDownMenuItem onClick={onQuarterlyReport}>
                    Insights Quarterly Report
                  </DropDownMenuItem>
                </DropDownMenu>
              </ClickOutDetector>
            </StyledLayout>

            <Button type={ButtonType.Alert} onClick={removeGameOpenModal} blurAfterClick>
              Remove
            </Button>
          </Layout>
        </TableCell>
      </TableRow>
      <ConfirmationDialog
        id={removeGameId}
        active={removeGameActive}
        closeModal={removeGameCloseModal}
        text={`Are you sure you want to remove ${props.game.name}?`}
        onClose={onRemove}
      />
    </>
  );
};

function lastQuarterWithAvailableOnePagerReport(date: Date): { year: number; quarter: number } {
  // One week ago, to make sure the report is available (that is how long it takes to produce the reports).
  const d = new Date(date.getTime()); // clone
  d.setDate(d.getDate() - 7);

  // zero-indexed quarter 0,1,2,3. Works as 1-indexed "previous quarter".
  const year = d.getFullYear();
  const monthsInQuarter = 3; // every quarter has 3 months
  const quarter = Math.floor(d.getMonth() / monthsInQuarter); // Note: month is zero-indexed too.
  if (quarter === 0) {
    return { year: year - 1, quarter: 4 };
  }
  return { year, quarter };
}
