import React, { useState, useCallback, useEffect } from 'react';
import { ModalId, Modal } from 'tachyon-modal';
import {
  ModalHeader,
  ModalBody,
  CoreText,
  Layout,
  Display,
  ModalFooter,
  FormGroup,
  Select,
} from 'twitch-core-ui';
import { RBACClient } from '~core/vienna';
import {
  GetUsersByCompanyIdRequest,
  Membership,
  Empty,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';
import './styles.scss';
import { useFormik } from 'formik';
import * as yup from 'yup';

export interface PublicProps {
  id: ModalId;
  closeModal: () => void;
  onCloseModal: (newRole: string, newOwner: string) => void;
  active: boolean;
  organizationId: string;
}

type Props = PublicProps;

export interface TransferOrganizationOwnershipForm {
  ownerNewRole: string;
  newOwnerId: string;
}

const validationSchema = yup.object<TransferOrganizationOwnershipForm>({
  ownerNewRole: yup.string().required(),
  newOwnerId: yup.string().required(),
});

export const TransferOrganizationModal: React.FC<Props> = (props) => {
  const { active, closeModal, organizationId } = props;
  const [memberships, setMemberships] = useState<Membership[]>([]);
  const [roles, setRoles] = useState<string[]>([]);
  const formik = useFormik<TransferOrganizationOwnershipForm>({
    initialValues: {
      ownerNewRole: '',
      newOwnerId: '',
    },
    onSubmit: handleSubmit,
    validationSchema,
  });

  async function handleSubmit(values: TransferOrganizationOwnershipForm) {
    props.onCloseModal(values.ownerNewRole, values.newOwnerId);
    props.closeModal();
  }

  async function getMembers() {
    if (!active) {
      return;
    }

    try {
      const result = await RBACClient.getUsersByCompanyId(
        new GetUsersByCompanyIdRequest({
          id: organizationId,
          limit: 1000,
        }),
      );
      setMemberships(result.memberships);
    } catch (err) {
      toast('Error fetching company members', { type: 'error' });
      closeModal();
    }
  }

  async function getRoles() {
    if (!active) {
      return;
    }
    try {
      const rolesResponse = await RBACClient.listAllRoles(new Empty());
      setRoles(rolesResponse.roles.filter((r) => r !== 'Owner'));
    } catch (err) {
      toast('Error fetching roles', { type: 'error' });
      closeModal();
    }
  }

  const getMembersCallback = useCallback(getMembers, [active, closeModal, organizationId]);
  const getRolesCallback = useCallback(getRoles, [active, closeModal]);

  useEffect(() => {
    getMembersCallback();
    getRolesCallback();
  }, [getMembersCallback, getRolesCallback, props.active]);

  const owner = memberships.find((m) => m.role === 'Owner');
  const otherMembers = memberships
    .filter((m) => m.role !== 'Owner')
    .sort((a, b) => a.lastName.localeCompare(b.lastName));

  if (!props.active) {
    return null;
  }

  return (
    <Modal id={props.id}>
      <ModalHeader
        title="Transfer Ownership"
        closeButton={{ onClick: props.closeModal, 'aria-label': 'Close' }}
      />
      <ModalBody>
        <form>
          <Layout className="transfer-organization-modal">
            <Layout display={Display.Flex} margin={{ bottom: 2 }}>
              <Layout margin={{ right: 1 }}>
                <CoreText>Current Owner:</CoreText>
              </Layout>
              <CoreText bold>
                {owner?.firstName} {owner?.lastName} ({owner?.twitchId})
              </CoreText>
            </Layout>
            <Layout>
              <Layout margin={{ bottom: 2 }}>
                <FormGroup label="New Owner">
                  <Select
                    value={formik.values.newOwnerId}
                    onChange={formik.handleChange}
                    onBlur={formik.handleBlur}
                    name="newOwnerId"
                  >
                    <option disabled value="">
                      Select a new Owner
                    </option>
                    {otherMembers.map((member) => (
                      <option key={member.twitchId} value={member.twitchId}>
                        {member.firstName} {member.lastName} ({member.twitchId})
                      </option>
                    ))}
                  </Select>
                </FormGroup>
              </Layout>
              <Layout>
                <FormGroup label="Owners New Role">
                  <Select
                    value={formik.values.ownerNewRole}
                    onChange={formik.handleChange}
                    name="ownerNewRole"
                    onBlur={formik.handleBlur}
                  >
                    <option disabled value="">
                      Select a role
                    </option>
                    {roles.map((role) => (
                      <option key={role} value={role}>
                        {role}
                      </option>
                    ))}
                  </Select>
                </FormGroup>
              </Layout>
            </Layout>
          </Layout>
        </form>
      </ModalBody>
      <ModalFooter
        primaryButtonProps={{
          onClick: (ev) => {
            ev.preventDefault();
            formik.submitForm();
          },
          children: 'Transfer',
          disabled: !formik.isValid || !formik.dirty,
        }}
        secondaryButtonProps={{ onClick: props.closeModal, children: 'Cancel' }}
      />
    </Modal>
  );
};
