import React, { useState, useCallback, useEffect } from 'react';
import { usePagination } from '~features/hooks/use-pagination';
import {
  AllowlistEntry,
  ListEntriesRequest,
  ListPagesRequest,
  Page,
  DeleteAllowlistEntryRequest,
  ListEntriesFilter,
  Product,
  UpdateAllowlistEntryRequest,
} from '~core/clients/twitch-e2-ingest-http/twitch.fulton.example';
import { TwitchE2IngestHTTPClient, RBACClient } from '~core/vienna';
import { EntityList } from '~features/entity-list';
import {
  E2ListItem,
  PublicProps as E2ListItemProps,
} from '~features/e2/list/components/e2-list-item/e2-list-item';
import { toast } from 'react-toastify';
import {
  SearchCompaniesRequest,
  SearchCompaniesResponse,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { DebounceTimer } from '~core/utils/debounce-timer';
import { useLazyQuery } from '@apollo/react-hooks';
import { SearchGames, SearchGamesVariables } from '~core/graphql/twitch-gql-schema';
import searchGameQuery from '~features/game-autocomplete/queries/search-game-query.gql';
import {
  Position,
  InjectLayout,
  Button,
  ButtonType,
  DropDownMenu,
  DropDownMenuItem,
  BalloonSize,
  Display,
  Layout,
  AlignSelf,
  FlexDirection,
  FormLabel,
} from 'twitch-core-ui';
import { Products } from '~features/e2/product';
import { E2ModalSchema } from '~features/allowlist-e2-modal';

const ENTRIES_LIMIT = 10;
const PAGE_0 = new Page();

const debounceTimer = new DebounceTimer(200);

export const E2ListPage: React.FC = () => {
  const { page, nextPage, previousPage, goToPage } = usePagination();
  const [entries, setEntries] = useState<AllowlistEntry[]>([]);
  const [total, setTotal] = useState(0);
  const [pages, setPages] = useState<Page[]>([PAGE_0]);
  const [loading, setLoading] = useState(false);

  const [searchTerm, setSearchTerm] = useState('');
  const [filter, setFilter] = useState<ListEntriesFilter>();
  const [isOpen, setIsOpen] = useState(false);

  const [searchCompanies, setSearchCompanies] = useState<Promise<SearchCompaniesResponse>>();
  const [searchGames] = useLazyQuery<SearchGames, SearchGamesVariables>(searchGameQuery, {
    onCompleted: (data) => {
      let gameIds: string[];
      if (data?.searchFor?.games?.items) {
        gameIds = data.searchFor.games.items.map((g) => g.id);
      }

      searchCompanies?.then((response) => {
        const companyIds = response.companies.map((c) => c.id);

        setFilter((prevFilter) => {
          const filter = new ListEntriesFilter(prevFilter);
          filter.gameIds = gameIds;
          filter.organizationIds = companyIds;
          filter.clientIds = [searchTerm];
          return filter;
        });

        // Return to the first page of search results.
        goToPage(1);
      });
    },
  });

  async function getEntries() {
    try {
      setLoading(true);
      // To get entries on page n, fetch the entries after page (n-1)'s cursor.
      const cursor = page <= 1 ? '' : pages[page - 1].endCursor;
      const response = await TwitchE2IngestHTTPClient.listEntries(
        new ListEntriesRequest({
          first: ENTRIES_LIMIT,
          after: cursor,
          filter: filter,
        }),
      );

      setEntries(response.entries);
      setLoading(false);
    } catch (err) {}
  }
  const getEntriesCallback = useCallback(getEntries, [page, pages, filter]);

  useEffect(() => {
    getEntriesCallback();
  }, [getEntriesCallback]);

  async function getPages() {
    try {
      setLoading(true);
      const response = await TwitchE2IngestHTTPClient.listPages(
        new ListPagesRequest({
          pageSize: ENTRIES_LIMIT,
          filter: filter,
        }),
      );

      setTotal(response.pages.length);
      setPages([PAGE_0, ...response.pages]);
      setLoading(false);
    } catch (err) {}
  }
  const getPagesCallback = useCallback(getPages, [filter]);

  useEffect(() => {
    getPagesCallback();
  }, [getPagesCallback]);

  async function deleteEntry(entry: AllowlistEntry) {
    try {
      await TwitchE2IngestHTTPClient.deleteAllowlistEntry(
        new DeleteAllowlistEntryRequest({
          gameId: entry.gameId,
          clientId: entry.clientId,
        }),
      );

      toast('Successfully deleted entry', { type: 'success' });
      getEntries();
    } catch (err) {
      toast(`Failed to delete entry: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function updateEntry(request: E2ModalSchema) {
    try {
      await TwitchE2IngestHTTPClient.updateAllowlistEntry(
        new UpdateAllowlistEntryRequest({ ...request }),
      );

      toast('Successfully updated entry', { type: 'success' });
      getEntries();
    } catch (err) {
      toast(`Failed to update entry: ${(err as Error).message}`, { type: 'error' });
    }
  }

  function onSearchUpdate(searchTerm: string) {
    setSearchTerm(searchTerm);

    if (searchTerm.length === 0) {
      setFilter((prevFilter) => {
        if (prevFilter) {
          return new ListEntriesFilter({ product: prevFilter.product });
        }
      });
      return;
    }

    if (searchTerm.length < 4) {
      return;
    }

    debounceTimer.invoke(() => {
      // Search for games that match the search term.
      // The query's `onComplete` callback will update the filter's gameIds.
      searchGames({ variables: { query: searchTerm } });

      // Search for companies that match the search term.
      const response = RBACClient.searchCompanies(
        new SearchCompaniesRequest({ query: searchTerm }),
      );
      setSearchCompanies(response);
    });
  }

  function setProductFilter(product: Product) {
    setIsOpen(false);
    setFilter((prevFilter) => {
      const filter = new ListEntriesFilter(prevFilter);
      filter.product = product;
      return filter;
    });
  }

  return (
    <EntityList<AllowlistEntry, E2ListItemProps>
      pagination={{
        onGoToPage: goToPage,
        onNextPage: nextPage,
        onPreviousPage: previousPage,
        totalPages: total,
        currentPage: page,
      }}
      loading={loading}
      items={entries}
      rowType={E2ListItem}
      rowProps={{
        onEditEntry: updateEntry,
        onDeleteEntry: deleteEntry,
        withOrganization: true,
      }}
      columns={[
        'Client ID',
        'Game ID',
        'Game Name',
        'Organization',
        'Product',
        'Date Created',
        'Created By',
        'Actions',
      ]}
      zeroStateMessage="No E2 clients found"
      header={{
        searchLabel: 'Filter by',
        searchTerm: searchTerm,
        onSearchUpdate: onSearchUpdate,
        searchPlaceholder: 'Search by client ID, organization or game',
        additionalFields: [
          <InjectLayout position={Position.Relative}>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
              <Layout display={Display.Flex} alignSelf={AlignSelf.Center} padding={{ right: 1 }}>
                <FormLabel id="product" label="Product" />
              </Layout>
              <Button
                type={ButtonType.Secondary}
                dropdown
                onClick={() => {
                  setIsOpen((open) => !open);
                }}
              >
                {Products.get(filter?.product || Product.UNKNOWN)}
              </Button>
              <DropDownMenu show={isOpen} size={BalloonSize.Small}>
                {Array.from(Products).map(([product, label]) => (
                  <DropDownMenuItem onClick={() => setProductFilter(product)} key={label}>
                    {label}
                  </DropDownMenuItem>
                ))}
              </DropDownMenu>
            </Layout>
          </InjectLayout>,
        ],
      }}
    />
  );
};
