import React from 'react';
import { ExtensionReviewPolicyCodes } from '~features/extensions/models/policy-codes';
import {
  Layout,
  Column,
  Grid,
  CheckBox,
  Input,
  InputType,
  Display,
  ButtonType,
  Button,
} from 'twitch-core-ui';

export enum PolicyAction {
  Removed = 'removed',
  Added = 'added',
}

export interface PublicProps {
  policies: string[];
  notes: string;
  onPoliciesUpdate: (policy: string, action: PolicyAction) => void;
  onNoteChange: (notes: string) => void;
  onApprove: () => void;
  onReject: () => void;
}

type Props = PublicProps;

export const ExtensionReviewForm: React.FC<Props> = (props) => {
  const policiesCount = ExtensionReviewPolicyCodes.length;
  const policiesPart1 = ExtensionReviewPolicyCodes.slice(0, policiesCount / 2);
  const policiesPart2 = ExtensionReviewPolicyCodes.slice(policiesCount / 2);

  function handlePolicyChange(event: React.ChangeEvent<HTMLInputElement>) {
    const checked = event.target.checked;
    props.onPoliciesUpdate(event.target.value, checked ? PolicyAction.Added : PolicyAction.Removed);
  }

  function renderCheckbox(policy: string, label: string) {
    let policyLabel = policy;
    if (label && label.length) {
      policyLabel += ` - ${label}`;
    }

    return (
      <CheckBox
        key={policy}
        label={policyLabel}
        value={policy}
        checked={props.policies.includes(policy)}
        onChange={handlePolicyChange}
      />
    );
  }

  return (
    <Layout>
      <Grid>
        <Column cols={{ default: 6, sm: 12, md: 6 }}>
          <Layout>{policiesPart1.map((policy) => renderCheckbox(policy.key, policy.label))}</Layout>
        </Column>
        <Column cols={{ default: 6, sm: 12, md: 6 }}>
          <Layout>{policiesPart2.map((policy) => renderCheckbox(policy.key, policy.label))}</Layout>
        </Column>
      </Grid>
      <Layout margin={{ bottom: 2 }}>
        <Input
          type={InputType.Text}
          onChange={(ev) => props.onNoteChange(ev.currentTarget.value)}
          value={props.notes}
          placeholder="General Notes"
        />
      </Layout>
      <Layout display={Display.Flex}>
        <Layout margin={{ right: 2 }}>
          <Button type={ButtonType.Alert} disabled={!props.notes.length} onClick={props.onReject}>
            Pending Action
          </Button>
        </Layout>
        <Button type={ButtonType.Success} disabled={!props.notes.length} onClick={props.onApprove}>
          Approve Extension
        </Button>
      </Layout>
    </Layout>
  );
};
