import React from 'react';
import { Extension } from '~core/clients/rbac/code.justin.tv.devrel';
import {
  Layout,
  StyledLayout,
  Background,
  Grid,
  Column,
  CoreLink,
  Display,
  Title,
  TitleSize,
  TableHeader,
  TableRow,
  Table,
  TableHeading,
  TableBody,
} from 'twitch-core-ui';
import { PageHeader } from '~features/page-header';
import { InfoField } from 'src/layout/info-field';
import { ExtensionReviewLogItem } from '~pages/extensions-review/components/extension-review-log-item';
import {
  ExtensionReviewForm,
  PolicyAction,
} from '~pages/extensions-review/components/extension-review-form';

export interface PublicProps {
  extension: Extension;
  reviewNotes: string;
  policies: string[];
  onReviewNotesChange: (notes: string) => void;
  onPolicyChange: (policy: string, action: PolicyAction) => void;
  onApprove: () => void;
  onReject: () => void;
}

type Props = PublicProps;

export const ExtensionReviewPresentation: React.FC<Props> = (props) => {
  const {
    policies,
    reviewNotes,
    extension,
    onReviewNotesChange,
    onPolicyChange,
    onApprove,
    onReject,
  } = props;
  const url = `https://www.twitch.tv/ext/${extension.id}-${extension.version}`;

  return (
    <Layout>
      <PageHeader title="Extension Review" />
      <Grid>
        <Column cols={{ default: 12, lg: 8, md: 10, sm: 12 }}>
          <Layout margin={{ bottom: 2 }}>
            <Title size={TitleSize.Small}>Basics</Title>
          </Layout>
          <StyledLayout background={Background.Base} padding={2}>
            <Layout display={Display.Flex}>
              <Layout margin={{ right: 3 }}>
                <InfoField title="Extension Name" value={extension.name} marginTop={0} />
              </Layout>
              <Layout margin={{ right: 3 }}>
                <InfoField title="Client ID" value={extension.id} />
              </Layout>
              <Layout margin={{ right: 3 }}>
                <InfoField title="Author" value={extension.authorName} />
              </Layout>
              <Layout margin={{ right: 3 }}>
                <InfoField title="Version" value={extension.version} />
              </Layout>
              <InfoField title="Current State" value={extension.state} />
            </Layout>

            <InfoField title="URL" value={<CoreLink linkTo={url}>{url}</CoreLink>} />
            <InfoField title="Summary" value={extension.summary} />
            <InfoField title="Description" value={extension.description} />
          </StyledLayout>
        </Column>
      </Grid>
      <Layout margin={{ y: 2 }}>
        <Layout margin={{ bottom: 2 }}>
          <Title size={TitleSize.Small}>Review Logs</Title>
        </Layout>
        <Table alternateRows>
          <TableHeader>
            <TableRow>
              <TableHeading label="Date" />
              <TableHeading label="Version" />
              <TableHeading label="State" />
              <TableHeading label="Author ID" />
              <TableHeading label="Notes" />
              <TableHeading label="Channel" />
              <TableHeading label="Reviewer ID" />
              <TableHeading label="Review Reason" />
              <TableHeading label="Reason Codes" />
              <TableHeading label="Salesforce Case ID" />
            </TableRow>
          </TableHeader>
          <TableBody>
            {extension.transitionLogs.map((log, index) => (
              <ExtensionReviewLogItem log={log} key={index} />
            ))}
          </TableBody>
        </Table>
      </Layout>
      <Layout>
        <Title size={TitleSize.Small}>Review</Title>
        <StyledLayout background={Background.Base} padding={2}>
          <ExtensionReviewForm
            policies={policies}
            notes={reviewNotes}
            onNoteChange={onReviewNotesChange}
            onPoliciesUpdate={onPolicyChange}
            onApprove={onApprove}
            onReject={onReject}
          />
        </StyledLayout>
      </Layout>
    </Layout>
  );
};
