import React, { useState, useCallback, useEffect } from 'react';
import { GamesReviewDetailPresentation } from '~pages/games-review-detail/games-review-detail-presentation';
import { useParams, useHistory } from 'react-router';
import getGameDetailsQuery from './queries/get-game-details.gql';
import { useQuery } from '@apollo/react-hooks';
import { getGameDetails, getGameDetailsVariables } from '~core/graphql/twitch-gql-schema';
import { Layout, Display, JustifyContent, LoadingSpinner, AlignItems } from 'twitch-core-ui';
import { RBACClient } from '~core/vienna';
import {
  GameApplication,
  Id,
  Company,
  ListCompaniesRequest,
  OnboardGameRequest,
  DeleteGameApplicationRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';

export interface RouteProps {
  gameApplicationId: string;
}

export const GamesReviewDetailPage: React.FC = () => {
  const { gameApplicationId } = useParams<RouteProps>();
  const history = useHistory();
  const [gameApplication, setGameApplication] = useState<GameApplication>();
  const [company, setCompany] = useState<Company>();
  const [owningCompany, setOwningCompany] = useState<Company>();

  async function getGameApplication() {
    try {
      const application = await RBACClient.getGameApplication(new Id({ id: gameApplicationId }));
      const companyResult = await RBACClient.getCompany(new Id({ id: application.companyId }));
      const owningCompanies = await RBACClient.listCompanies(
        new ListCompaniesRequest({
          ownsGameId: application.gameId,
          limit: 1,
          offset: 0,
        }),
      );
      if (owningCompanies?.companies.length) {
        setOwningCompany(owningCompanies.companies[0]);
      }
      setGameApplication(application);
      setCompany(companyResult);
    } catch {
      toast('Failed to load Game Application', { type: 'error' });
    }
  }
  const getGameApplicationCallback = useCallback(getGameApplication, [gameApplicationId]);
  useEffect(() => {
    getGameApplicationCallback();
  }, [getGameApplicationCallback, gameApplicationId]);

  const { data, loading } = useQuery<getGameDetails, getGameDetailsVariables>(getGameDetailsQuery, {
    variables: { id: gameApplication?.gameId?.toString() || '' },
    skip: !gameApplication,
  });

  async function approveApplication() {
    try {
      await RBACClient.onboardGame(
        new OnboardGameRequest({
          gameApplicationId: gameApplicationId,
        }),
      );
      toast('Successfully approved the application', { type: 'success' });
      history.push('/games/reviews');
    } catch (err) {
      toast(`Error while approving application: ${(err as Error).message}`, {
        type: 'error',
      });
    }
  }

  async function rejectApplication() {
    try {
      await RBACClient.deleteGameApplication(
        new DeleteGameApplicationRequest({
          id: gameApplicationId,
          skipEmail: false,
        }),
      );
      toast('Successfully rejected the application', { type: 'success' });
      history.push('/games/reviews');
    } catch (err) {
      toast(`Error while rejecting application: ${(err as Error).message}`, {
        type: 'error',
      });
    }
  }

  if (!gameApplication || !company || loading || !data || !data.game) {
    return (
      <Layout
        fullHeight
        fullWidth
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        alignItems={AlignItems.Center}
      >
        <LoadingSpinner />
      </Layout>
    );
  }

  return (
    <GamesReviewDetailPresentation
      game={data.game}
      application={gameApplication}
      company={company}
      owningCompany={owningCompany}
      onApprove={approveApplication}
      onReject={rejectApplication}
    />
  );
};
