import React, { useState, useCallback, useEffect } from 'react';
import { useParams } from 'react-router';
import { RBACClient } from '~core/vienna';
import {
  GetDeveloperApplicationsRequest,
  DeveloperApplication,
  OnboardDeveloperRequest,
  DeleteDeveloperApplicationRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { toast } from 'react-toastify';
import { OrganizationDevelopersPresentation } from '~pages/organization-developers/organization-developers-presentation';

interface RouterProps {
  organizationId: string;
}

export const OrganizationDevelopersPage: React.FC = () => {
  const { organizationId } = useParams<RouterProps>();
  const [developers, setDevelopers] = useState<DeveloperApplication[]>([]);

  async function loadDevelopers() {
    try {
      const result = await RBACClient.getDeveloperApplications(
        new GetDeveloperApplicationsRequest({
          companyId: organizationId,
        }),
      );
      setDevelopers(result.developerApplications);
    } catch (err) {
      toast(`Failed to load Developers: ${(err as Error).message}`);
    }
  }

  async function onboardDeveloper(applicationId: string, role: string) {
    try {
      await RBACClient.onboardDeveloper(
        new OnboardDeveloperRequest({
          developerApplicationId: applicationId,
          role: role,
          roleId: '',
        }),
      );
      toast('Onboarded Developer!', { type: 'success' });
      loadDevelopers();
    } catch (err) {
      toast(`Error onboarding Developer: ${(err as Error).message}`, { type: 'error' });
    }
  }

  async function removeApplication(applicationId: string) {
    try {
      await RBACClient.deleteDeveloperApplication(
        new DeleteDeveloperApplicationRequest({
          id: applicationId,
          skipEmail: true,
        }),
      );
      toast('Deleted Application', { type: 'success' });
      loadDevelopers();
    } catch (err) {
      toast(`Error deleting application: ${(err as Error).message}`, { type: 'error' });
    }
  }

  const loadDevelopersCallback = useCallback(loadDevelopers, [organizationId]);

  useEffect(() => {
    loadDevelopersCallback();
  }, [loadDevelopersCallback]);

  return (
    <OrganizationDevelopersPresentation
      developers={developers}
      onOnboarding={onboardDeveloper}
      onRemove={removeApplication}
    />
  );
};
